// This file is part of the Java Card Firewall Tester program.
// 
// Authors: 
// 
// Wojciech Mostowski, woj@cs.ru.nl
// Erik Poll, erikpoll@cs.ru.nl
// Radboud University Nijmegen
// The Netherlands
// 
// Copyright (c) Wojciech Mostowski, Erik Poll,
// Radboud University Nijmegen (RU),
// Stichting Technische Wetenschappen (STW)
// 
// The Java Card Firewall Tester has been developed for the PinPas Java
// Card project, see http://www.win.tue.nl/pinpasjc/. The program is
// distributed under the licence terms that can be found in the LICENCE
// file in the main installation directory. Please refer to the LICENCE &
// README files for further information.

package firewallclients;

import javacard.framework.*;
import javacard.security.*;
import javacardx.crypto.*;
import firewallinterface.*;

public class ClientApplet1 extends Applet {

    /** The different AID that we are going to use in the test. */
    static final byte[] serverAID = { (byte)'f', (byte)'s', (byte)'e', (byte)'r', (byte)'v', (byte)'e', (byte)'r', (byte)'.', (byte)'a', (byte)'p', (byte)'p' };
    static final byte[] serverAIDms = { (byte)'f', (byte)'s', (byte)'e', (byte)'r', (byte)'v', (byte)'e', (byte)'r', (byte)'m', (byte)'s', (byte)'.', (byte)'a', (byte)'p', (byte)'p' };
    static final byte[] c1AID = { (byte)'f', (byte)'c', (byte)'l', (byte)'i', (byte)'e', (byte)'n', (byte)'t', (byte)'.', (byte)'a', (byte)'p', (byte)'p', (byte)'1' };
    static final byte[] c2AID = { (byte)'f', (byte)'c', (byte)'l', (byte)'i', (byte)'e', (byte)'n', (byte)'t', (byte)'.', (byte)'a', (byte)'p', (byte)'p', (byte)'2' };

    private static final short MAX_TESTS = 30;

    /** Test numbers */
    private static final byte TEST1 = 0x01;
    private static final byte TEST2 = 0x02;
    private static final byte TEST3 = 0x03;
    private static final byte TEST4 = 0x04;
    private static final byte TEST5 = 0x05;
    private static final byte TEST6 = 0x06;
    private static final byte TEST7 = 0x07;
    private static final byte TEST8 = 0x08;
    private static final byte TEST9 = 0x09;
    private static final byte TESTA = 0x0a;
    private static final byte TESTB = 0x0b;
    private static final byte TESTC = 0x0c;
    private static final byte TESTD = 0x0d;
    private static final byte TESTE = 0x0e;
    private static final byte TESTF = 0x0f;
    private static final byte TEST10 = 0x10;
    private static final byte TEST11 = 0x11;
    private static final byte TEST12 = 0x12;
    private static final byte TEST13 = 0x13;
    private static final byte TEST14 = 0x14;
    private static final byte TEST15 = 0x15;

    /** An APDU command to reset the references stored in the library, necessary to clean
      * inter-applet dependencies, so that the applets can be removed.
      */
    private static final byte RESET = (byte)0xaa;


    private APDU tmpAPDU = null;
    private static APDU stmpAPDU = null;
    private Exception tmpException = null;
    private static Exception stmpException = null;
    private Cipher cipher = null;

    private byte[] persistentArray = null;
    private byte[] resetArray = null;
    private byte[] deselectArray = null;

    private static byte[] staticArray = null;

    private byte installStoreCode1 = 0;
    private byte installStoreCode2 = 0;

    private byte[] resultCodes = null;
    private short resultLength = 0;

    private AID testAID = null;
    private AID installAID = null;
    private static AID stmpAID = null;
    private TestInterface1 ti = null;

    private short tmp = 0;
    private boolean tmpBool = false;
    private TestObjectInterface tmpTestObjectInterface;
    private TestObject tmpTestObject;
    private TestObjectShareable tmpTestObjectShareable;
    private TestInterface2 tmpTestInterface2;

    private TestInterface1 getTestInterface1() {
        if (ti == null) {
            try {
                AID a = JCSystem.lookupAID(serverAID, TestInterface1.ZERO, (byte)serverAID.length);
                ti = (TestInterface1)JCSystem.getAppletShareableInterfaceObject(a, (byte)0);
            }catch (SecurityException se) {
                ti = null;
            }
        }
        return ti;
    }

    private TestInterface1 getTestInterface1Multiselectable() {
        if (ti == null) {
            try {
                AID a = JCSystem.lookupAID(serverAIDms, TestInterface1.ZERO, (byte)serverAIDms.length);
                ti = (TestInterface1)JCSystem.getAppletShareableInterfaceObject(a, (byte)0);
            }catch (SecurityException se) {
                ti = null;
            }
        }
        return ti;
    }

    public static void install(byte[] bArray, short bOffset, byte bLength) {
        new ClientApplet1(bArray).register(bArray, (short) (bOffset + 1), bArray[bOffset]);
    }

    /** @description Perform some test already during installation, e.g.
      * test that the install byte array is not storable.
      */
    private ClientApplet1(byte[] bArray) {
        resultCodes = new byte[MAX_TESTS];
        try {
            persistentArray = bArray;
            installStoreCode1 = 0x00;
        }catch (SecurityException se) {
            installStoreCode1 = 0x01;
        }catch (Throwable t) {
            installStoreCode1 = decodeExceptionCode(t);
        }

        try {
            staticArray = bArray;
            installStoreCode2 = 0x00;
        }catch (SecurityException se) {
            installStoreCode2 = 0x01;
        }catch (Throwable t) {
            installStoreCode2 = decodeExceptionCode(t);
        }

        persistentArray = new byte[TestInterface1.TESTAIDLEN];
        testAID = new AID(persistentArray, TestInterface1.ZERO, TestInterface1.TESTAIDLEN);
        installAID = JCSystem.getAID();
        ShareableObject.createTestObjectClient();
        ShareableObject.createShareableTestObjectClient();
    }

    public void process(APDU apdu) {
        if (selectingApplet()) {
            return;
        }
        byte[] buf = apdu.getBuffer();
        apdu.setIncomingAndReceive();
        Util.arrayFillNonAtomic(resultCodes, TestInterface1.ZERO, (short)resultCodes.length, (byte)0);
        resultLength = 0;
        switch (buf[ISO7816.OFFSET_INS]) {
        case TEST1:
            doTest1(apdu);
            sendResult(apdu);
            return;
        case TEST2:
            doTest2(apdu);
            sendResult(apdu);
            return;
        case TEST3:
            doTest3(apdu);
            sendResult(apdu);
            return;
        case TEST4:
            doTest4(apdu);
            sendResult(apdu);
            return;
        case TEST5:
            doTest5(apdu);
            sendResult(apdu);
            return;
        case TEST6:
            doTest6(apdu);
            sendResult(apdu);
            return;
        case TEST7:
            doTest7(apdu);
            sendResult(apdu);
            return;
        case TEST8:
            doTest8(apdu);
            sendResult(apdu);
            return;
        case TEST9:
            doTest9(apdu);
            sendResult(apdu);
            return;
        case TESTA:
            doTestA(apdu);
            sendResult(apdu);
            return;
        case TESTB:
            doTestB(apdu);
            sendResult(apdu);
            return;
        case TESTC:
            doTestC(apdu);
            sendResult(apdu);
            return;
        case TESTD:
            doTestD(apdu);
            sendResult(apdu);
            return;
        case TESTE:
            doTestE(apdu);
            sendResult(apdu);
            return;
        case TESTF:
            doTestF(apdu);
            sendResult(apdu);
            return;
        case TEST10:
            doTest10(apdu);
            sendResult(apdu);
            return;
        case TEST11:
            doTest11(apdu);
            sendResult(apdu);
            return;
        case TEST12:
            doTest12(apdu);
            sendResult(apdu);
            return;
        case TEST13:
            doTest13(apdu);
            sendResult(apdu);
            return;
        case TEST14:
            doTest10a(apdu);
            sendResult(apdu);
            return;
        case TEST15:
            doTest11a(apdu);
            sendResult(apdu);
            return;
        case RESET:
            ShareableObject.resetReferences();
            return;
        default:
            ISOException.throwIt(ISO7816.SW_INS_NOT_SUPPORTED);
        }
    }


    /** @description Test for the storability of JCRE objects: APDU instance,
      * APDU buffer, installation bArray, system owned AIDs, system owned exceptions
      */
    private void doTest1(APDU apdu) {
        resultCodes[resultLength++] = installStoreCode1;
        resultCodes[resultLength++] = installStoreCode2;

        try {
            tmpAPDU = apdu;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            stmpAPDU = apdu;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            byte[] t = persistentArray;
            persistentArray = apdu.getBuffer();
            persistentArray[0] = 0x01;
            persistentArray = t;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            staticArray = apdu.getBuffer();
            staticArray[0] = 0x01;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            try {
                ISOException.throwIt(ISO7816.SW_WRONG_LENGTH);
            }catch (ISOException ie) {
                tmpException = ie;
            }
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            try {
                ISOException.throwIt(ISO7816.SW_WRONG_LENGTH);
            }catch (ISOException ie) {
                stmpException = ie;
            }
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            AID a = testAID;
            testAID = JCSystem.getAID();
            testAID.equals(null);
            testAID = a;
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            stmpAID = JCSystem.getAID();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description Test the behaviour of JCSystem.getAID()
      * and JCSystem.getPreviousContextAID().
      */
    private void doTest2(APDU apdu) {
        AID a = JCSystem.getAID();
        if (a.equals(c1AID, TestInterface1.ZERO, (byte)c1AID.length))
            resultCodes[resultLength++] = 0x01;
        if (ClientApplet2.instance.equalsAID(c2AID))
            resultCodes[resultLength++] = 0x01;
        if (ClientApplet2.instance.checkPreviousAID())
            resultCodes[resultLength++] = 0x01;
        TestInterface1 ti = getTestInterface1();
        byte[] b = apdu.getBuffer();
        Util.arrayCopyNonAtomic(serverAID, TestInterface1.ZERO, b, TestInterface1.ZERO, (short)serverAID.length);
        if (ti.checkAID(b, (byte)serverAID.length))
            resultCodes[resultLength++] = 0x01;
        Util.arrayCopyNonAtomic(c1AID, TestInterface1.ZERO, b, TestInterface1.ZERO, (short)c1AID.length);
        if (ti.checkPreviousAID(b, (byte)c1AID.length))
            resultCodes[resultLength++] = 0x01;
        if (JCSystem.getPreviousContextAID() == null)
            resultCodes[resultLength++] = 0x01;
        if (installAID == null)
            resultCodes[resultLength++] = 0x01;
    }

    /** @description Test the behaviour of CLEAR_ON_DESELECT arrays (Section 6.1.5),
      * the same context, different applets
      */
    private void doTest3(APDU apdu) {
        try {
            if (resetArray == null)
                resetArray = ClientApplet2.instance.createArray(TestInterface1.SIZE, JCSystem.CLEAR_ON_RESET);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
        try {
            ClientApplet2.instance.readWriteArray(resetArray);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
        try {
            if (deselectArray == null)
                deselectArray = ClientApplet2.instance.createArray(TestInterface1.SIZE, JCSystem.CLEAR_ON_DESELECT);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
        try {
            ClientApplet2.instance.readWriteArray(deselectArray);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description Test the behaviour of CLEAR_ON_RESET and CLEAR_ON_DESELECT
      * arrays (Section 6.1.5), different contexts. This test should also be
      * run with the multiselectable server applet selected on another logical channel
      * on cards that support logical channels and multiselectable applets.
      */
    private void doTest4(APDU apdu) {

//    TestInterface1 ti = getTestInterface1();
        /*JC22*/
        TestInterface1 ti = getTestInterface1Multiselectable();

        try {
            ti.createArray((short)1, JCSystem.NOT_A_TRANSIENT_OBJECT);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ti.createArray((short)1, JCSystem.CLEAR_ON_RESET);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ti.createArray((short)1, JCSystem.CLEAR_ON_DESELECT);
        }catch (SystemException se) {
            resultCodes[resultLength] = (byte)(se.getReason() == SystemException.ILLEGAL_TRANSIENT ? 0x01 :
                                               decodeExceptionCode(se));
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ti.readWriteArray(JCSystem.NOT_A_TRANSIENT_OBJECT);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ti.readWriteArray(JCSystem.CLEAR_ON_RESET);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ti.readWriteArray(JCSystem.CLEAR_ON_DESELECT);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description Test the behaviour of static access outside of the current
      * context in a library package (there should be no context switch)
      */
    private void doTest5(APDU apdu) {

        try {
            resultCodes[resultLength] =
                (byte)(TestObject.checkAIDs(
                           JCSystem.getAID(), JCSystem.getPreviousContextAID()) ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength ++;

        try {
            short v = TestObject.ps;
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength ++;
    }

    /** @description Test the behaviour of static access outside of the current
      * context in an applet package. For this test the server export file
      * modification is needed, currently we skip it.
      */
    private void doTest6(APDU apdu) {
        resultCodes[resultLength++] = 0x01;
    }

    /** @description Test access to JCRE entry points and global arrays:
      * AID (permanent), APDU object (temporary), APDU buffer (global)
      */
    private void doTest7(APDU apdu) {
        TestInterface1 ti = getTestInterface1();
        try {
            ti.accessAPDU(apdu);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ti.accessGlobalArray(apdu.getBuffer());
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        // Applet owned AID
        try {
            ti.accessAID(testAID);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        // System owned AID
        try {
            ti.accessAID(JCSystem.getAID());
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description Checking of access rules on parameters by (TP)EP objects
      */
    private void doTest8(APDU apdu) {
        TestInterface1 ti = getTestInterface1();
        for (short i = 0; i < TestInterface1.AIDBYTEMETHODS; i++) {
            // Java Card 2.1.* does not have the getPartialBytes method
// if(i == 2) { resultCodes[resultLength++] = 0x01; continue; }
            try {
                ti.parameterCheckAID(persistentArray, i);
                resultCodes[resultLength] = 0x00;
            }catch (SecurityException se) {
                resultCodes[resultLength] = 0x01;
            }catch (Throwable t) {
                resultCodes[resultLength] = decodeExceptionCode(t);
            }
            resultLength++;
        }

        try {
            tmpBool = ti.parameterCheckAID2(this);
            resultCodes[resultLength] = 0x00;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        // Applet owned AIDs
        try {
            tmpBool = ti.parameterCheckAID2(testAID);
            resultCodes[resultLength] = 0x00;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = ti.parameterCheckAID3(testAID);
            resultCodes[resultLength] = 0x00;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        // System owned AIDs
        try {
            tmpBool = ti.parameterCheckAID2(JCSystem.getAID());
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = ti.parameterCheckAID3(JCSystem.getAID());
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

    }

    /** @description JCRE cannnot read CLEAR_ON_DESELECT arrays if
      * the currently active context is not the one of the currently selected
      * applet, it can read CLEAR_ON_RESET arrays
      */
    private void doTest9(APDU apdu) {
        TestInterface1 ti = getTestInterface1();

        for (short i = 0; i < TestInterface1.AIDBYTEMETHODS; i++) {
            // Java Card 2.1.* does not have the getPartialBytes method
// if(i == 2) { resultCodes[resultLength++] = 0x01; continue;}
            try {
                ti.parameterCheckAID(ti.getArray(JCSystem.CLEAR_ON_DESELECT), i);
            }catch (SecurityException se) {
                resultCodes[resultLength] = 0x01;
            }catch (Throwable t) {
                resultCodes[resultLength] = decodeExceptionCode(t);
            }
            resultLength++;
        }

        for (short i = 0; i < TestInterface1.AIDBYTEMETHODS; i++) {
            // Java Card 2.1.* does not have the getPartialBytes method
// if(i == 2) { resultCodes[resultLength++] = 0x01; continue; }
            try {
                ti.parameterCheckAID(ti.getArray(JCSystem.CLEAR_ON_RESET), i);
                resultCodes[resultLength] = 0x01;
            }catch (Throwable t) {
                resultCodes[resultLength] = decodeExceptionCode(t);
            }
            resultLength++;
        }
    }

    /** @description JCRE can read CLEAR_ON_DESELECT arrays if the currently active
      * context is the one of the currently selected applet
      */
    private void doTestA(APDU apdu) {
        for (short i = 0; i < TestInterface1.AIDBYTEMETHODS; i++) {
            // Java Card 2.1.* does not have the getPartialBytes method
// if(i == 2) { resultCodes[resultLength++] = 0x01; continue; }
            try {
                parameterCheckAID(deselectArray, i);
                resultCodes[resultLength] = 0x01;
            }catch (Throwable t) {
                resultCodes[resultLength] = decodeExceptionCode(t);
            }
            resultLength++;
        }
    }

    /** @description static byte array belongs to the package
      */
    private void doTestB(APDU apdu) {
        try {
            byte[] a = ClientApplet2.staticArray;
            a[1] = (byte)(a[0] + 1);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description method calls and field access within package and accross firewall
      */
    private void doTestC(APDU apdu) {
        try {
            tmp = ShareableObject.getTestObjectServer().a;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getTestObjectServer().a = 2;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getTestObjectServer().setA();
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmp = ShareableObject.getShareableTestObjectServer().a;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getShareableTestObjectServer().a = 2;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getShareableTestObjectServer().setA();
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getShareableTestObjectServer().setB();
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ((TestInterface2)ShareableObject.getShareableTestObjectServer()).setB();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmp = ShareableObject.getTestObjectClient().a;
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getTestObjectClient().a = 2;
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getTestObjectClient().setA();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmp = ShareableObject.getShareableTestObjectClient().a;
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getShareableTestObjectClient().a = 2;
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ShareableObject.getShareableTestObjectClient().setA();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description check the possiblity to do casts and instanceof checks
      */
    private void doTestD(APDU apdu) {
        try {
            tmpTestObjectInterface = (TestObjectInterface)(Object)ShareableObject.getTestObjectClient();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpTestObject = (TestObject)ShareableObject.getTestObjectClient();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getTestObjectClient() instanceof TestObject);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getTestObjectClient() instanceof Shareable);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x00 : 0x01);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpTestObject = (TestObject)(Object)ShareableObject.getTestObjectServer();
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getTestObjectServer() instanceof TestObject);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getTestObjectServer() instanceof Shareable);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpTestObjectInterface = (TestObjectInterface)(Object)ShareableObject.getShareableTestObjectClient();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpTestObjectShareable = (TestObjectShareable)ShareableObject.getShareableTestObjectClient();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpTestInterface2 = (TestInterface2)ShareableObject.getShareableTestObjectClient();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getShareableTestObjectClient() instanceof TestObjectShareable);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getShareableTestObjectClient() instanceof Shareable);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getShareableTestObjectClient() instanceof TestInterface2);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpTestObjectInterface = (TestObjectInterface)(Object)ShareableObject.getShareableTestObjectServer();
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpTestObjectShareable = (TestObjectShareable)(Object)ShareableObject.getShareableTestObjectServer();
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpTestInterface2 = (TestInterface2)ShareableObject.getShareableTestObjectServer();
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getShareableTestObjectServer() instanceof TestObjectShareable);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getShareableTestObjectServer() instanceof Shareable);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (ShareableObject.getShareableTestObjectServer() instanceof TestInterface2);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            Object a = JCSystem.getAID();
            tmpBool = (a instanceof Shareable);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x00 : 0x01);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmpBool = (JCSystem.getAID() instanceof AID);
            resultCodes[resultLength] = (byte)(tmpBool ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            try {
                ISOException.throwIt(ISO7816.SW_WRONG_LENGTH);
            } catch (ISOException ie) {
                tmpBool = (ie instanceof ISOException);
            }
            resultCodes[resultLength] = (byte)(tmpBool ? 0x01 : 0x00);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            try {
                ISOException.throwIt(ISO7816.SW_WRONG_LENGTH);
            } catch (ISOException ie) {
                tmpBool = (ie instanceof Shareable);
            }
            resultCodes[resultLength] = (byte)(tmpBool ? 0x00 : 0x01);
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description check the access to context owned arrays accross contexts
      */
    private void doTestE(APDU apdu) {
        TestInterface1 ti = getTestInterface1();

        try {
            byte r = ti.checkArray1(persistentArray);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            ti.checkArray2(persistentArray);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            tmp = ti.checkArray3(persistentArray);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description exception throwing/owning, context restoration
      */
    private void doTestF(APDU apdu) {
        TestInterface1 ti = getTestInterface1();

        try{
            ti.throwException1(new NullPointerException());
        }catch (NullPointerException npe) {
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try{
            try {
                ISOException.throwIt(ISO7816.SW_WRONG_LENGTH);
            }catch (ISOException ie) {
                ti.throwException2(ie);
            }
        }catch (ISOException e) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        AID a1 = JCSystem.getAID();
        try{
            ti.throwException3();
        }catch (Throwable t) {
            AID a2 = JCSystem.getAID();
            resultCodes[resultLength++] = (byte)(a1.equals(a2) ? 0x01 : 0x00);
        }

        a1 = JCSystem.getPreviousContextAID();
        byte result = 0;
        try{
            ti.throwException3();
        }catch (Throwable t) {
            AID a2 = JCSystem.getPreviousContextAID();
            if (a1 == null && a2 == null)
                result = 0x01;
            else
                result = (byte)(a1.equals(a2) ? 0x01 : 0x00);
            resultCodes[resultLength++] = result;
        }
    }

    /** @description accessing shareable interface objects when the
      * non-multiselectable server is active on another logical channel.
      */
    private void doTest10(APDU apdu) {
        TestInterface1 ti = getTestInterface1();

        // The server should be selected on another channel
        try {
            ti.getArray(JCSystem.NOT_A_TRANSIENT_OBJECT);
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description accessing shareable interface objects when the
      * multiselectable server is active on another logical channel.
      */
    private void doTest10a(APDU apdu) {
        TestInterface1 ti = getTestInterface1Multiselectable();

        // The server should be selected on another channel
        try {
            ti.getArray(JCSystem.NOT_A_TRANSIENT_OBJECT);
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description acquiring shareable interface objects when the
      * non-multiselectable server is active on another logical channel.
      */
    private void doTest11(APDU apdu) {
        AID a = JCSystem.lookupAID(serverAID, TestInterface1.ZERO, (byte)serverAID.length);
        TestInterface1 tmp = ti;
        // The server should be selected on another channel
        try {
            ti = (TestInterface1)JCSystem.getAppletShareableInterfaceObject(a, (byte)0);
            ti = tmp;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description acquiring shareable interface objects when the
      * multiselectable server is active on another logical channel.
      */
    private void doTest11a(APDU apdu) {
        AID a = JCSystem.lookupAID(serverAIDms, TestInterface1.ZERO, (byte)serverAIDms.length);
        TestInterface1 tmp = ti;
        // The server should be selected on another channel
        try {
            ti = (TestInterface1)JCSystem.getAppletShareableInterfaceObject(a, (byte)0);
            resultCodes[resultLength] = 0x01;
            ti = tmp;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }

    /** @description test context restoration on stack overflow
      */
    private void doTest12(APDU apdu) {
        TestInterface1 ti = getTestInterface1();
        AID a1 = JCSystem.getAID();
        try {
            ti.callRecursive();
        }catch (Throwable t) {
            AID a2 = JCSystem.getAID();
            resultCodes[resultLength++] = (byte)(a1.equals(a2) ? 0x01 : decodeExceptionCode(t));
        }
    }

    /** @description External ciphers can be used externally.
      */
    private void doTest13(APDU apdu) {
        TestInterface1 ti = getTestInterface1();

        try {
            byte[] buf = apdu.getBuffer();
            Util.arrayFillNonAtomic(buf, TestInterface1.ZERO, (short)8, (byte)0x02);
            ti.accessCipher(buf, false);
            resultCodes[resultLength] = 0x00;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x01;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;

        try {
            byte[] buf = apdu.getBuffer();
            Util.arrayFillNonAtomic(buf, TestInterface1.ZERO, (short)8, (byte)0x02);
            ti.accessCipher(buf, true);
            resultCodes[resultLength] = 0x01;
        }catch (SecurityException se) {
            resultCodes[resultLength] = 0x00;
        }catch (Throwable t) {
            resultCodes[resultLength] = decodeExceptionCode(t);
        }
        resultLength++;
    }


    private void parameterCheckAID(byte[] array, short mnum) {
        AID a = JCSystem.getAID();
        switch (mnum) {
        case 0:
        {
            boolean b = a.equals(array, TestInterface1.ZERO, (byte)5);
        }
        return;
        case 1:
        {
            byte b = a.getBytes(array, TestInterface1.ZERO);
        }
        return;
// Java Card 2.1.* does not have the getPartialBytes method
        /*JC22*/
        case 2:
        /*JC22*/         {
            byte b = a.getPartialBytes(TestInterface1.ZERO, array, TestInterface1.ZERO, (byte)5);
        }
        /*JC22*/         return;
        case 3:
        {
            boolean b = a.partialEquals(array, TestInterface1.ZERO, (byte)5);
        }
        return;
        case 4:
            a = new AID(array, TestInterface1.ZERO, (byte)array.length);
            return;
        default:
            return;
        }
    }

    private static byte decodeExceptionCode(Throwable t) {
        if (t instanceof ArithmeticException)                   return 0x21;
        else if (t instanceof ArrayStoreException)             return 0x22;
        else if (t instanceof ClassCastException)              return 0x23;
        else if (t instanceof ArrayIndexOutOfBoundsException)  return 0x24;
        else if (t instanceof IndexOutOfBoundsException)       return 0x25;
        else if (t instanceof NegativeArraySizeException)      return 0x26;
        else if (t instanceof NullPointerException)            return 0x27;
        else if (t instanceof SecurityException)               return 0x28;
        /*JC22*/
        else if (t instanceof java.rmi.RemoteException)        return 0x31;
        /*JC22*/
        else if (t instanceof UserException)                   return 0x41;
        else if (t instanceof APDUException)                   return 0x51;
        else if (t instanceof javacard.security.CryptoException)    return 0x52;
        else if (t instanceof ISOException)                    return 0x53;
        else if (t instanceof PINException)                    return 0x54;
        /*JC22*/
        else if (t instanceof javacard.framework.service.ServiceException)   return 0x55;
        else if (t instanceof SystemException)                 return 0x56;
        else if (t instanceof TransactionException)            return 0x57;
        else return (byte)0xAA;
    }

    private void sendResult(APDU apdu) {
        byte[] buf = apdu.getBuffer();
        Util.arrayCopyNonAtomic(resultCodes, TestInterface1.ZERO, buf, TestInterface1.ZERO, resultLength);
        try {
            apdu.setOutgoing();
        }catch (APDUException ae) {
        }
        apdu.setOutgoingLength(resultLength);
        apdu.sendBytes(TestInterface1.ZERO, resultLength);
    }

}

