// This file is part of the Java Card Firewall Tester program.
// 
// Authors: 
// 
// Wojciech Mostowski, woj@cs.ru.nl
// Erik Poll, erikpoll@cs.ru.nl
// Radboud University Nijmegen
// The Netherlands
// 
// Copyright (c) Wojciech Mostowski, Erik Poll,
// Radboud University Nijmegen (RU),
// Stichting Technische Wetenschappen (STW)
// 
// The Java Card Firewall Tester has been developed for the PinPas Java
// Card project, see http://www.win.tue.nl/pinpasjc/. The program is
// distributed under the licence terms that can be found in the LICENCE
// file in the main installation directory. Please refer to the LICENCE &
// README files for further information.

package sos.smartcards.firewallhost;

import java.io.*;
import java.util.*;
import javax.smartcardio.*;


public class FirewallHost {

    public static final byte[] fclient = "fclient.app1".getBytes();
    public static final byte[] fserver = "fserver.app".getBytes();
    public static final byte[] fserverms = "fserverms.app".getBytes();
   
    private static void usage() {
	System.out.println("Usage:");
	System.out.println("   java sos.smartcards.firewallhost.FirewallHost"+
			   " [ -jc21 | -reset ]");
	System.out.println("");
	System.out.println("   -jc21   indicates that the test card is only Java Card 2.1.*");
	System.out.println("   -reset  resets inter-applet dependencies on the card for applet removal");
	System.out.println();
	System.exit(0);
    }

    public static void main(String[] args) throws Exception{

	int rIndex = 0;
	boolean jc21 = false;
	boolean reset = false;
	if(args.length > 0) {
	    int l = args.length;
	    int i = 0;
	    while(i < l) {
		if(args[i].equals("-jc21")) {
		    jc21 = true;
		} else if(args[i].equals("-reset")) {
		    reset = true;
		}else {
		    usage();
		}
		i++;
	    }
	} 
	try {

	    TerminalFactory tf = TerminalFactory.getInstance("PC/SC", null);
	    CardChannel channel = null;
	    CardTerminals terminals = tf.terminals();
	    Card card = null;
	    boolean cardPresent = false;
	    while(!cardPresent) {
		for (CardTerminal terminal : terminals.list(CardTerminals.State.CARD_PRESENT)) {
		    try {
			card = terminal.connect("*");
			card.beginExclusive();
			ATR a = card.getATR();
			channel = card.getBasicChannel();
			cardPresent = true;
		    }catch(CardException ce) {
			System.out.println("Couldn't esatablish the connection.");
			ce.printStackTrace();
		    }
		}
		if(!cardPresent) {
		    System.out.println("Waiting for a card...");
		    terminals.waitForChange(0);
		}
	    }

	    if(reset) {
	      byte[][] aids = new byte[][]{ fclient, fserver, fserverms };
	      CommandAPDU resetCommand = new CommandAPDU(new byte[] {00, (byte)0xAA, 0, 0, 0});
	      reset = false;
	      for(byte[] aid : aids) {
	        try {
		  selectAID(channel, aid);
		  int sw = channel.transmit(resetCommand).getSW();
		  if(sw == 0x9000) {
		    reset = true;
		    break;
		  }
		}catch(CardException ce) {
		}
	      }
	      if(reset) {
	        System.out.println("References reset successfully, you can remove the applets.");
	      }else{
	        System.out.println("Could not reset references...");
	      }
	    }else{
	      TestSuite ts = new TestSuite();
	      try {
		selectAID(channel, fclient);
		ts.runTests(card, channel, fserver, fserverms, jc21);
		System.out.println("Tests finished.");
	      }catch(Exception jce) {
		jce.printStackTrace();
	      }
	    }

	}catch(Exception e) { 
	    System.out.println("Failed: " + e);
	}

    }

    public static byte[] selectAID(CardChannel channel, byte[] aid) throws CardException {
        CommandAPDU cmd = new CommandAPDU(
					  (byte)0, (byte)0xA4,
					  (byte) 0x04, (byte) 0x00, aid);
        ResponseAPDU resp = channel.transmit(cmd);
        if(resp.getSW() != 0x9000) {
            throw new CardException("Could not select applet "+byteArrayToString(aid));
        }
        return resp.getBytes();
    }

    public static String byteArrayToString(byte[] a) {
	String result = "";
	String onebyte = null;
	for(int i=0; i< a.length; i++) {
	    onebyte = Integer.toHexString(a[i]);
	    if(onebyte.length() == 1)
		onebyte = "0" + onebyte;
	    else
		onebyte = onebyte.substring(onebyte.length()-2);
	    result = result + onebyte.toUpperCase() + " ";
	}
	return result;
    }


}
