;; --------------------------------------------------
;; Top-down inference algorithm, implemented in CLIPS.
;; Principles of Intelligent Systems, pp. 86-95
;; --------------------------------------------------

(deftemplate goal
  "goal or subgoal variable"
  (slot name))

(deftemplate fact
  "member of set of facts"
  (slot variable)
  (slot value))

(deftemplate prompt
  "prompt concerning a variable with legal values"
  (slot variable)
  (slot text
    (type STRING))
  (multislot legal))

(defrule Infer
  "infer variable and select applicable rule"
  (goal (name ?variable))
  (if ? ?new-variable $? then ? ?variable ?value)
  =>
  (assert (goal (name ?new-variable))))

(defrule GoalTraced?
  "goal has been traced"
  ?f <- (goal (name ?variable))
  (fact (variable ?variable)
        (value ?value))
  =>
  (retract ?f)
  (format t "Traced: %s is %s%n" ?variable ?value))

(deffunction Error ( )
  (format t "Illegal predicate or action%n"))

(deffunction ExecPred (?pred ?v ?w)
  (switch ?pred
    (case same then (eq ?v ?w))
    (case notsame then (neq ?v ?w))
    (case equal then (= ?v ?w))
    (case lessthan then (< ?v ?w))
    (case greaterthan then (> ?v ?w))
    (default (Error))))

(defrule EvalConditions-1
  "check condition, and delete rule when it has failed"
  (fact (variable ?variable) (value ?value1))
  ?f <- (if ?predicate ?variable ?value2 ? ? ? $?)
  (not (test (ExecPred ?predicate ?value1 ?value2)))
  =>
  (retract ?f))

(defrule EvalConditions-2
  "if a condition succeeds, evaluate next condition"
  (fact (variable ?variable) (value ?value1))
  ?f <- (if ?predicate ?variable ?value2 and $?rest then 
            ?action ?concl-var ?concl-value)
  (test (ExecPred ?predicate ?value1 ?value2))
  =>
  (retract ?f)
  (assert (if $?rest then ?action ?concl-var ?concl-value)))

(deffunction ExecAction (?act ?var ?val)
  (switch ?act
    (case add then (assert (fact (variable ?var) (value ?val))))
    (case write then (printout t ?var " is " ?val))
    (default (Error))))

(defrule EvalConclusion
  "rule satisfied, therefore delete rule and evaluate conclusion"
  (fact (variable ?variable) (value ?value1))
  ?f <- (if ?predicate ?variable ?value2 then 
            ?action ?concl-variable ?concl-value)
  (test (ExecPred ?predicate ?value1 ?value2))
  =>
  (retract ?f)
  (ExecAction ?action ?concl-variable ?concl-value))

(defrule Ask
  "ask value of variable to user"
  ?f1 <- (goal (name ?variable))
  ?f2 <- (prompt (variable ?variable) (text ?text &: (neq ?text ""))
                 (legal $?answers))
  =>
  (retract ?f1)
  (format t "%s " ?text)
  (printout t ?answers " ")
  (bind ?reply (read))
  (if (member (lowcase ?reply) ?answers) 
      then (assert (fact (variable ?variable)
                         (value ?reply)))
           (retract ?f2)
      else (assert (goal (name ?variable)))))


;; ----------------------
;; Example Knowledge Base
;; ----------------------

(deffacts knowledge-base 
  (goal (name z))
  
  (prompt (variable x) (text "Enter x.") (legal small large))
  (prompt (variable y) (text "Enter y.") (legal small large))
  (prompt (variable z) (legal yes no))
  (prompt (variable u) (text "Enter u.") (legal red green))
  (prompt (variable v) (text "Enter v.") (legal red green))
  (prompt (variable w) (text "Enter w.") (legal small large))

;; rulebase

  (if same x small and same y large
   then add w yes)
  (if same u green and same v red
   then add w no)
  (if same w yes 
   then add z yes)
  (if same w no 
   then add z no)

)
