import java.awt.Color;
import java.util.Stack;

/**
 * A pen can be used to draw on a canvas. The pen maintains a position, direction, color,
 * and an up/down state. The pen can be moved across the canvas. If the pen is down, it 
 * leaves a line on the canvas when moved. (If it is up, it will not draw a line.)
 * 
 * @author Michael Kölling & David J. Barnes (Changed bij Sjaak Smetsers)
 * @version 2015.06.08
 */
public class Pen
{
    
    private int xPosition;
    private int yPosition;
    private int rotation;
    private Color color;
    private boolean penDown;

    private Canvas canvas;


    /**
     * Create a new Pen for a given canvas. The initial color is black, and the pen is down.
     * The position of the pen is given w.r.t. the origin located in the lower left
     * corner of the window.
     *
     * @param xPos  the initial horizontal coordinate of the pen
     * @param yPos  the initial vertical coordinate of the pen
     * @param rotation  the initial rotation of the pen
     * @param drawingCanvas  the canvas to draw on
     */
    public Pen( int xPos, int yPos, int rotation, Canvas drawingCanvas ) {
        this.xPosition = xPos;
        this.yPosition = yPos;
        this.rotation  = rotation;
        this.penDown   = true;
        this.color     = Color.BLACK;
        this.canvas    = drawingCanvas;
    }

    /**
     * Move the specified distance in the current direction. If the pen is down, 
     * leave a line on the canvas.
     * 
     * @param distance  The distance to move forward from the current location.
     */
    public void move(int distance)
    {
        double angle = Math.toRadians(rotation);
        int newX = (int) Math.round(xPosition + Math.cos(angle) * distance);
        int newY = (int) Math.round(yPosition + Math.sin(angle) * distance);
        
        moveTo(newX, newY);
    }

    /**
     * Move to the specified location. If the pen is down, leave a line on the canvas.
     * 
     * @param x   The x-coordinate to move to.
     * @param y   The y-coordinate to move to.
     */
    public void moveTo(int x, int y)
    {
        if (penDown) {
            canvas.setForegroundColor(color);
            int canvasHeight = (int) canvas.getSize().getHeight();
            canvas.drawLine(xPosition, canvasHeight - yPosition, x, canvasHeight - y);
        }

        xPosition = x;
        yPosition = y;
    }

    /**
     * Turn the specified amount (out of a 360 degree circle) clockwise from the current 
     * rotation.
     * 
     * @param degrees  The amount of degrees to turn. (360 is a full circle.)
     */
    public void turn(int degrees)
    {
        rotation = rotation + degrees;
    }

    /**
     * Turn to the specified direction. 0 is right, 90 is down, 180 is left, 270 is up.
     * 
     * @param angle  The angle to turn to.
     */
    public void turnTo(int angle)
    {
        rotation = angle;
    }

    /**
     * Set the drawing color.
     * 
     * @param newColor  The color to use for subsequent drawing operations.
     */
    public void setColor(Color newColor)
    {
        color = newColor;
    }

    /**
     * Lift the pen up. Moving afterwards will not leave a line on the canvas.
     */
    public void penUp()
    {
        penDown = false;
    }

    /**
     * Put the pen down. Moving afterwards will leave a line on the canvas.
     */
    public void penDown()
    {
        penDown = true;
    }

    private static class PositionAndDirection {
        int x, y, rotation;
        
        private PositionAndDirection (int x, int y, int rotation) {
            this.x = x; this.y = y; this.rotation = rotation;
        }
    }
    
    private Stack<PositionAndDirection> penStates = new Stack<> ();

    public void save () {
        penStates.push( new PositionAndDirection ( xPosition, yPosition, rotation ) );
    }

    public void restore () {
        PositionAndDirection p_and_d = penStates.pop();
        xPosition = p_and_d.x;
        yPosition = p_and_d.y;
        rotation  = p_and_d.rotation;
    }

}
