/**************************************************************************************************
 Copyright 2019--2025 Cynthia Kop

 Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 in compliance with the License.
 You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software distributed under the
 License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 express or implied.
 See the License for the specific language governing permissions and limitations under the License.
 *************************************************************************************************/

package charlie.terms;

import org.junit.jupiter.api.Test;
import static org.junit.jupiter.api.Assertions.*;
import java.util.List;
import java.util.ArrayList;
import java.util.TreeMap;
import java.util.TreeSet;

import charlie.util.Pair;
import charlie.util.NullStorageException;
import charlie.types.Type;
import charlie.terms.position.*;
import charlie.terms.replaceable.ReplaceableList;

public class ApplicationTest extends TermTestFoundation {
  @Test
  public void testUnaryWithNullArg() {
    Variable head = new Binder("x", arrowType("a", "b"));
    Term arg = null;
    assertThrows(NullStorageException.class, () ->new Application(head, arg));
  }

  @Test
  public void testBinaryWithNullHead() {
    assertThrows(NullStorageException.class, () ->
      new Application(null, constantTerm("a", baseType("b")),
                            constantTerm("a", baseType("c"))));
  }

  @Test
  public void testNullArgs() {
    Constant f = new Constant("f", arrowType("a", "b"));
    List<Term> args = null;
    assertThrows(NullStorageException.class, () -> new Application(f, args));
  }

  @Test
  public void testTooManyArgs() {
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    Variable x = new Binder("x", type);
    List<Term> args = new ArrayList<Term>();
    args.add(constantTerm("c", baseType("a")));
    args.add(constantTerm("d", baseType("b")));
    args.add(constantTerm("e", baseType("a")));
    assertThrows(TypingException.class, () -> new Application(x, args));
  }

  @Test
  public void testTooManyArgsToApplication() {
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    Variable x = new Binder("x", type);
    Term head = x.apply(constantTerm("c", baseType("a")));
    List<Term> args = new ArrayList<Term>();
    args.add(constantTerm("d", baseType("b")));
    args.add(constantTerm("e", baseType("a")));
    assertThrows(TypingException.class, () -> new Application(head, args));
  }

  @Test
  public void testConstructorBadArgType() {
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    Term head = constantTerm("f", type);
    List<Term> args = new ArrayList<Term>();
    args.add(constantTerm("c", baseType("a")));
    args.add(constantTerm("d", baseType("a")));
    assertThrows(TypingException.class, () -> new Application(head, args));
  }

  @Test
  public void testConstructorBadArgTypeToApplication() {
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    Term head = constantTerm("f", type).apply(constantTerm("c", baseType("a")));
    assertThrows(TypingException.class, () ->
      new Application(head, constantTerm("d", baseType("a"))));
  }

  @Test
  public void testCreateEmptyApplication() {
    Term head = new Var("x", arrowType(baseType("a"), arrowType("b", "a")));
    List<Term> args = new ArrayList<Term>();
    assertThrows(IllegalArgumentException.class, () -> new Application(head, args));
  }

  @Test
  public void testConstructApplicationFromApplicationHead() {
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    Term head = constantTerm("f", type).apply(constantTerm("c", baseType("a")));
    Term t = new Application(head, constantTerm("d", baseType("b")));
    assertTrue(t.toString().equals("f(c, d)"));
    assertTrue(t.queryType().equals(baseType("a")));

    Term s = TermFactory.createApp(t, new ArrayList<Term>());
    assertTrue(s.equals(t));
  }

  @Test
  public void testFunctionalTermBasics() {
    Term t = twoArgFuncTerm();
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    assertTrue(t.isApplication());
    assertTrue(t.isApplicative());
    assertTrue(t.isFunctionalTerm());
    assertFalse(t.isConstant());
    assertFalse(t.isVariable());
    assertFalse(t.isVarTerm());
    assertTrue(t.isPattern());
    assertTrue(t.isApplication());
    assertTrue(t.queryRoot().equals(new Constant("f", type)));
    assertTrue(t.queryHead().equals(t.queryRoot()));
    assertTrue(t.queryHead().queryType().toString().equals("a → b → a"));
    assertTrue(t.queryType().equals(baseType("a")));
    assertTrue(t.isClosed());
    assertTrue(t.isGround());
    assertTrue(t.isTrueTerm());
    assertTrue(t.toString().equals("f(c, g(d))"));
    Term q = null;
    assertFalse(t.equals(q));
  }

  @Test
  public void testVarTermBasics() {
    Term t = twoArgVarTerm();
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    assertTrue(t.isApplication());
    assertTrue(t.isApplicative());
    assertTrue(t.isVarTerm());
    assertFalse(t.isConstant());
    assertFalse(t.isVariable());
    assertFalse(t.isFunctionalTerm());
    assertFalse(t.isPattern());
    assertTrue(t.isApplication());
    assertTrue(t.queryVariable().toString().equals("x"));
    assertTrue(t.queryHead().equals(t.queryVariable()));
    assertTrue(t.queryHead().queryType().toString().equals("a → b → a"));
    assertTrue(t.queryType().equals(baseType("a")));
    assertTrue(t.toString().equals("x(c, g(y))"));
    assertTrue(t.isClosed());
    assertFalse(t.isGround());
    assertTrue(t.isTrueTerm());
    assertTrue(t.isLinear());
  }

  @Test
  public void testLinearity() {
    Type oo = arrowType("o", "o");
    Term f = constantTerm("f", arrowType(baseType("o"), oo));
    Term g = constantTerm("g", arrowType(oo, arrowType(oo, baseType("o"))));
    Term h = constantTerm("h", arrowType(baseType("o"), arrowType(baseType("o"), oo)));
    Variable x = TermFactory.createVar("x", baseType("o"));
    Variable y = TermFactory.createBinder("y", baseType("o"));
    Variable z = TermFactory.createBinder("z", baseType("o"));
    MetaVariable zz = TermFactory.createMetaVar("Z", arrowType("o", "o"), 1);
    MetaVariable hh = TermFactory.createMetaVar("h", arrowType("o", "o"), 1);
    Term zy = TermFactory.createMeta(zz, y);
    Term hy = TermFactory.createMeta(hh, y);
    Term hz = TermFactory.createMeta(hh, z);
    // f(x, x)
    Term fxx = new Application(f, x, x);
    assertFalse(fxx.isLinear());
    // g(λz.H[z], λy.H[y])
    Term ghh = new Application(g, new Abstraction(z, hz), new Abstraction(y, hy));
    assertFalse(ghh.isLinear());
    // λy.h(Z[y], y, H[y])
    Term hzh = new Abstraction(y, (new Application(h, zy, y)).apply(hy));
    assertTrue(hzh.isLinear());
  }

  @Test
  public void testStoreFunctionSymbols() {
    Term t = twoArgFuncTerm();
    TreeSet<FunctionSymbol> set = new TreeSet<FunctionSymbol>();
    set.add(new Constant("c", baseType("a")));
    set.add(new Constant("f", baseType("b")));
    t.storeFunctionSymbols(set);
    assertTrue(set.size() == 5);
    assertTrue(set.toString().equals("[c, d, f, f, g]"));
  }

  @Test
  public void testTheory() {
    // +(x(0), y(z))
    Term zero = new IntegerValue(0);
    Type i = zero.queryType();
    Term x = new Var("x", arrowType(i, i));
    Term y = new Binder("y", i);
    Term t = new Application(TheoryFactory.plusSymbol, x.apply(zero), y);
    assertTrue(t.isTheoryTerm());
    assertFalse(t.isValue());
    assertTrue(t.toValue() == null);
    // z(0) with z :: Int → a
    Var z = new Var("z", arrowType(i, baseType("a")));
    t = z.apply(zero);
    assertFalse(t.isTheoryTerm());
    // +(1, 2)
    t = new Application(TheoryFactory.plusSymbol, new IntegerValue(1), new IntegerValue(2));
    assertTrue(t.isTheoryTerm());
    assertFalse(t.isValue());
    assertTrue(t.toValue() == null);
  }

  @Test
  public void testAbstractionTermBasics() {
    Variable x = new Binder("x", baseType("o"));
    Term abs = new Abstraction(x, x);
    Term t = new Application(abs, constantTerm("a", baseType("o")));
    assertTrue(t.isApplication());
    assertFalse(t.isApplicative());
    assertFalse(t.isVarTerm());
    assertFalse(t.isFunctionalTerm());
    assertFalse(t.isConstant());
    assertFalse(t.isVariable());
    assertFalse(t.isPattern());
    assertTrue(t.queryVariable() == x);
    assertTrue(t.queryHead() == abs);
    assertTrue(t.queryType().toString().equals("o"));
    assertTrue(t.toString().equals("(λx.x)(a)"));
  }

  @Test
  public void testPatternWithAbstractionBasics() {
    // x(y, λz.f(z))
    Variable x = new Binder("x", arrowType(baseType("A"), arrowType(
      arrowType("B", "A"), baseType("B"))));
    Variable y = new Var("y", baseType("A"));
    Variable z = new Binder("z", baseType("B"));
    Constant f = new Constant("f", arrowType("B", "A"));
    Term abs = new Abstraction(z, new Application(f, z));
    Term t = new Application(x, y, abs);

    assertTrue(t.isApplication());
    assertFalse(t.isApplicative());
    assertTrue(t.isPattern());
    assertTrue(t.isVarTerm());
    assertTrue(t.queryVariable() == x);
    assertTrue(t.queryHead() == x);
    assertTrue(t.queryType().equals(baseType("B")));
    assertTrue(t.toString().equals("x(y, λz.f(z))"));
  }

  @Test
  public void testIncorrectSubterm() {
    Term t = twoArgVarTerm();
    assertThrows(IndexOutOfBoundsException.class, () -> t.queryArgument(0));
    assertThrows(IndexOutOfBoundsException.class, () -> t.queryArgument(3));
  }

  @Test
  public void testArguments() {
    Term t = twoArgFuncTerm();
    assertTrue(t.numberArguments() == 2);
    assertTrue(t.queryArgument(1).equals(constantTerm("c", baseType("a"))));
    assertTrue(t.queryArgument(2).toString().equals("g(d)"));
    List<Term> args = t.queryArguments();
    assertTrue(args.get(0) == t.queryArgument(1));
    assertTrue(args.get(1) == t.queryArgument(2));
    assertTrue(args.size() == 2);
  }

  @Test
  public void testMetaArguments() {
    // z⟨a,b⟩(c,d,e)
    Type o = baseType("o");
    Type type = arrowType(o, arrowType(o, arrowType(o, arrowType(o, arrowType(o, o)))));
    MetaVariable z = TermFactory.createMetaVar("Z", type, 2);
    Term zab = TermFactory.createMeta(z, constantTerm("a", o), constantTerm("b", o));
    Term term =
      zab.apply(constantTerm("c", o)).apply(constantTerm("d", o)).apply(constantTerm("e", o));
    assertTrue(term.numberArguments() == 3);
    assertTrue(term.numberMetaArguments() == 2);
    assertTrue(term.queryMetaArgument(1).equals(constantTerm("a", o)));
    assertTrue(term.queryMetaArgument(2).equals(constantTerm("b", o)));
    assertTrue(term.queryArgument(1).equals(constantTerm("c", o)));
  }

  @Test
  public void testImmediateHeadSubterms() {
    Term t = twoArgVarTerm();
    assertTrue(t.queryImmediateHeadSubterm(0).toString().equals("x"));
    assertTrue(t.queryImmediateHeadSubterm(1).toString().equals("x(c)"));
    assertTrue(t.queryImmediateHeadSubterm(2).toString().equals("x(c, g(y))"));
  }

  @Test
  public void testInappropriateRootRequest() {
    Term t = twoArgVarTerm();
    assertThrows(InappropriatePatternDataException.class, () -> t.queryRoot());
  }

  @Test
  public void testInappropriateVariableRequest() {
    Term t = twoArgFuncTerm();
    assertThrows(InappropriatePatternDataException.class, () -> t.queryVariable());
  }

  @Test
  public void testInappropriateAbstractionSubtermRequest() {
    Term t = twoArgFuncTerm();
    assertThrows(InappropriatePatternDataException.class, () -> t.queryAbstractionSubterm());
  }

  @Test
  public void testGoodAbstractionSubtermRequest() {
    Variable x = new Binder("x", baseType("o"));
    Term abs = new Abstraction(x, x);
    Term term = new Application(abs, constantTerm("a", baseType("o")));
    assertTrue(term.queryAbstractionSubterm() == x);
  }

  @Test
  public void testFirstOrderFunctionalTerm() {
    Type aa = arrowType("a", "a");
    Term s = twoArgFuncTerm();
    Term t = unaryTerm("h", aa, new Var("x", baseType("b")));
    Type utype = arrowType(baseType("a"), arrowType(aa, baseType("b")));
    Term q = new Application(new Constant("u", utype), s, t); 
    assertTrue(s.isFirstOrder());
    assertFalse(t.isFirstOrder());
    assertFalse(q.isFirstOrder());
  }

  @Test
  public void testFirstOrderVarTerm() {
    Variable y = new Binder("y", arrowType("o", "o"));
    Term x3 = new Application(y, constantTerm("c", baseType("o")));
    assertFalse(x3.isFirstOrder());
  }

  @Test
  public void testNonPatternDueToVariableApplication() {
    Variable x = new Var("x", arrowType("A", "B"));
    Term xa = new Application(x, constantTerm("a", baseType("A")));
    Term f = new Constant("f", arrowType("B", "B"));
    Term fxa = new Application(f, xa);
    assertFalse(fxa.isPattern());
    assertTrue(fxa.isSemiPattern());
  }

  @Test
  public void testBinderPattern() {
    Variable x = new Binder("x", arrowType(baseType("b"), arrowType("b", "a")));
    Variable y = new Var("y", baseType("b"));
    Variable z = new Var("z", arrowType("b", "b"));
    Term ba = new Constant("bb", arrowType("a", "b")).apply(constantTerm("aa", baseType("a")));
    List<Term> args = new ArrayList<Term>();
    args.add(y);
    args.add(ba);
    Term xybterm = new Application(x, args);  // x(y, bb(aa))
    assertTrue(xybterm.isPattern());    // we're allowed to apply binder variables
    assertTrue(xybterm.isSemiPattern());
    args.set(1, z.apply(ba));
    Term combiterm = new Application(x, args);  // x(y, bb(aa), z(bb(aa)))
    assertFalse(combiterm.isPattern()); // but the arguments do all need to be patterns :)
  }

  @Test
  public void testSemiNonPattern() {
    MetaVariable z = TermFactory.createMetaVar("Z", baseType("o"), arrowType("o", "o"));
    MetaVariable y = TermFactory.createMetaVar("Z", baseType("o"), arrowType("o", "o"));
    Term u = new Constant("u", baseType("o"));
    Term v = new Constant("v", arrowType(arrowType("o", "o"), baseType("o")));
    Variable x = new Binder("x", baseType("o"));
    Term zx = TermFactory.createMeta(z, x);
    Term zu = TermFactory.createMeta(z, u);
    // z[x](u)
    Term zxu = zx.apply(u);
    assertFalse(zxu.isPattern());
    assertTrue(zxu.isSemiPattern());
    // z[u](x)
    Term zux = zu.apply(x);
    assertFalse(zux.isPattern());
    assertFalse(zux.isSemiPattern());
    // v(z[x])
    Term vzx = v.apply(zx);
    assertTrue(vzx.isPattern());
    assertTrue(vzx.isSemiPattern());
    // v(z[u])
    Term vzu = v.apply(zu);
    assertFalse(vzu.isPattern());
    assertFalse(vzu.isSemiPattern());
  }

  @Test
  public void testSubterms() {
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    Variable z = new Binder("Z", type);
    Term arg1 = unaryTerm("g", baseType("a"), new Var("x", baseType("b")));
    Term arg2 = constantTerm("c", baseType("b"));
    Term term = new Application(z, arg1, arg2);    // Z(g(x),c)
    List<Pair<Term,Position>> lst = term.querySubterms();
    assertTrue(lst.size() == 4);
    assertTrue(lst.get(0).snd().toString().equals("1.1"));
    assertTrue(lst.get(0).fst().toString().equals("x"));
    assertTrue(lst.get(1).snd().toString().equals("1"));
    assertTrue(lst.get(1).fst() == arg1);
    assertTrue(lst.get(2).snd().toString().equals("2"));
    assertTrue(lst.get(2).fst() == term.queryArgument(2));
    assertTrue(lst.get(3).snd().toString().equals("ε"));
    assertTrue(lst.get(3).fst() == term);
  }

  @Test
  public void testVisitor() {
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    Variable z = new Binder("Z", type);
    Term arg1 = unaryTerm("g", baseType("a"), new Var("x", baseType("b")));
    Term arg2 = constantTerm("c", baseType("b"));
    Application term = new Application(z, arg1, arg2);    // Z(g(x),c)
    ArrayList<String> parts = new ArrayList<String>();
    term.visitSubterms( (s,p) -> parts.add(s.toString()));
    assertTrue(parts.size() == 4);
    assertTrue(parts.get(0).equals("x"));
    assertTrue(parts.get(1).equals("g(x)"));
    assertTrue(parts.get(2).equals("c"));
    assertTrue(parts.get(3).equals("Z(g(x), c)"));
  }

  @Test
  public void testFullPositionsForBetaRedex() {
    Variable x = new Binder("x", baseType("A"));
    Constant a = new Constant("a", baseType("A"));
    Constant b = new Constant("b", baseType("B"));
    Constant f = new Constant("f", arrowType(baseType("A"), arrowType("B", "C")));
    Term term = new Application(new Abstraction(x, f.apply(x)), a, b); // (λx.f(x))(a, b)
    List<Position> lst = term.queryPositions(false);
    assertTrue(lst.size() == 5);
    assertTrue(lst.get(0).toString().equals("0.1"));
    assertTrue(lst.get(1).toString().equals("0"));
    assertTrue(lst.get(2).toString().equals("1"));
    assertTrue(lst.get(3).toString().equals("2"));
    assertTrue(lst.get(4).toString().equals("ε"));
  }

  @Test
  public void testPartialPositions() {
    Type type = arrowType(baseType("a"), arrowType("b", "a"));
    Variable z = new Binder("Z", type);
    Term arg1 = unaryTerm("g", baseType("a"), new Var("x", baseType("b")));
    Term arg2 = constantTerm("c", baseType("b"));
    Term term = new Application(z, arg1, arg2);    // Z(g(x),c)
    List<Position> lst = term.queryPositions(true);
    assertTrue(lst.size() == 7);
    assertTrue(lst.get(0).toString().equals("1.1"));
    assertTrue(lst.get(1).toString().equals("1.☆1"));
    assertTrue(lst.get(2).toString().equals("1"));
    assertTrue(lst.get(3).toString().equals("2"));
    assertTrue(lst.get(4).toString().equals("☆2"));
    assertTrue(lst.get(5).toString().equals("☆1"));
    assertTrue(lst.get(6).toString().equals("ε"));
  }

  @Test
  public void testFreeReplaceables() {
    // let's create: Z(Z(x,h(λz.c(z))),g(J[y],x)), where Z, x and y are variables and J is a
    // meta-variable
    Variable z = new Binder("Z", arrowType(baseType("a"),arrowType("b","a")));
    FunctionSymbol g = new Constant("g", arrowType(baseType("b"),arrowType("a","b")));
    FunctionSymbol c = new Constant("c", arrowType("o", "o"));
    FunctionSymbol h = new Constant("h", arrowType(arrowType("o", "o"), baseType("b")));
    Variable x = new Binder("x", baseType("a"));
    Variable y = new Var("y", baseType("b"));
    Variable z2 = new Binder("z", baseType("o"));
    MetaVariable j = TermFactory.createMetaVar("J", arrowType("b", "b"), 1);
    Term jy = TermFactory.createMeta(j, y);
    Term hlambdazcz = new Application(h, new Abstraction(z2, c.apply(z2)));
    Term s = new Application(z, new Application(z, x, hlambdazcz), new Application(g, jy, x));
    ReplaceableList lst = s.freeReplaceables();
    assertTrue(lst.contains(x));
    assertTrue(lst.contains(y));
    assertTrue(lst.contains(z));
    assertTrue(lst.contains(j));
    assertTrue(lst.size() == 4);
  }

  @Test
  public void testFreeReplaceablesReuse() {
    // let's create: f(g(x), h(y,y))
    Variable x = new Var("x", baseType("o"));
    Variable y = new Var("x", baseType("o"));
    Term gx = unaryTerm("g", baseType("o"), x);
    Term hyy = TermFactory.createConstant("h", 2).apply(y).apply(y);
    Term term = TermFactory.createConstant("f", 2).apply(gx).apply(hyy);
    assertTrue(gx.freeReplaceables() == x.freeReplaceables());
    assertTrue(hyy.freeReplaceables() == y.freeReplaceables());
    assertTrue(term.freeReplaceables().size() == 2);
  }

  @Test
  public void testBoundVars() {
    // let's create: f(λx.Z(x), Y, g(λz.z, λx,u.h(x,y)))
    Variable x = new Binder("x", baseType("o"));
    Variable y = new Binder("y", baseType("o"));
    Variable z = new Binder("z", baseType("o"));
    Variable u = new Binder("u", baseType("o"));
    Variable bZ = new Var("Z", arrowType("o", "o"));
    Variable bY = new Var("Y", baseType("o"));
    FunctionSymbol h = new Constant("h", arrowType(baseType("o"), arrowType("o", "o")));
    FunctionSymbol g = new Constant("g", arrowType(arrowType("o", "o"),
      arrowType(arrowType(baseType("o"), arrowType("o", "o")), baseType("o"))));
    FunctionSymbol f = new Constant("f",
      arrowType(arrowType("o", "o"), arrowType(baseType("o"), arrowType("o", "o"))));
    Term ahxy = new Abstraction(x, new Abstraction(u, new Application(h, x, y)));
    Term az = new Abstraction(z, z);
    Term gterm = new Application(g, az, ahxy);
    Term aZx = new Abstraction(x, new Application(bZ, x));
    Term fterm = new Application(f, aZx, bY).apply(gterm);

    ReplaceableList freeVars = fterm.freeReplaceables();
    ReplaceableList boundVars = fterm.boundVars();
    assertTrue(freeVars.size() == 3);
    assertTrue(boundVars.size() == 3);
    assertTrue(boundVars.contains(x));
    assertTrue(boundVars.contains(z));
    assertTrue(boundVars.contains(u));
    assertTrue(freeVars.contains(y));
    assertTrue(freeVars.contains(bY));
    assertTrue(freeVars.contains(bZ));
  }

  @Test
  public void testBoundVarsReuse() {
    Variable x = new Binder("x", baseType("o"));
    Variable y = new Binder("y", baseType("o"));
    Variable bY = new Var("Y", baseType("o"));
    Type oo = arrowType("o", "o");
    Constant f = new Constant("f", arrowType(baseType("o"), oo));
    Constant g = new Constant("g", arrowType(oo, baseType("o")));
    Constant h = new Constant("h", arrowType(baseType("o"), arrowType(oo, oo)));
    Constant i = new Constant("i", arrowType(oo, arrowType(oo, oo)));
    Constant b = new Constant("B", baseType("o"));
    // let's create: (λxy.f(x,y))(g(λx.x), g(λy.y))
    Term ax = new Abstraction(x, x);
    Term gx = g.apply(ax);
    Term gy = g.apply(new Abstraction(y, y));
    Term head = new Abstraction(x, new Abstraction(y, new Application(f, x, y)));
    Term term1 = new Application(head, gx, gy);
    assertTrue(ax.boundVars() == gx.boundVars());
    assertTrue(term1.boundVars() == head.boundVars());
    // let's create: h(Y, λx.x, B)
    Term term2 = (new Application(h, bY, ax)).apply(b);
    assertTrue(term2.boundVars() == ax.boundVars());
    // let's create: i(λx.x, λy.f(g(λx.x), y))
    Term fterm = new Application(f, gx, y);
    Term afterm = new Abstraction(y, fterm);
    Term term3 = new Application(i, ax, afterm);
    assertTrue(term3.boundVars() == afterm.boundVars());
    assertFalse(afterm.boundVars() == fterm.boundVars());
  }

  @Test
  public void testMVars() {
    // let's create: a(f(Z[X], X), Y[b])
    Variable a = new Binder("a", arrowType(baseType("o"), arrowType("o", "o")));
    Variable b = new Binder("b", baseType("o"));
    Var x = new Var("Y", baseType("o"));
    MetaVariable y = TermFactory.createMetaVar("Y", arrowType("o", "o"), 1);
    MetaVariable z = TermFactory.createMetaVar("Z", arrowType("o", "o"), 1);
    Term zx = TermFactory.createMeta(z, x);
    Term f = constantTerm("f", arrowType(baseType("o"), arrowType("o", "o")));
    Term fzxx = TermFactory.createApp(f, zx, x);
    Term yb = TermFactory.createMeta(y, b);
    Term term = TermFactory.createApp(a, fzxx, yb);
    // let's see if all the mvars are as expected!
    Environment<MetaVariable> mvars = term.mvars();
    assertTrue(mvars.size() == 3);
    assertTrue(mvars.contains(x));
    assertTrue(mvars.contains(y));
    assertTrue(mvars.contains(z));
    mvars = fzxx.mvars();
    assertTrue(mvars.size() == 2);
    assertTrue(mvars.contains(x));
    assertFalse(mvars.contains(y));
    assertTrue(mvars.contains(z));
    mvars = yb.mvars();
    assertTrue(mvars.size() == 1);
  }

  @Test
  public void testFullSubtermGood() {
    Position p;
    Term s = twoArgFuncTerm();
    p = Position.empty;
    assertTrue(s.querySubterm(p).equals(s));
    p = new ArgumentPos(1, Position.empty);
    assertTrue(s.querySubterm(p).equals(constantTerm("c", baseType("a"))));
    p = new ArgumentPos(2, new ArgumentPos(1, Position.empty));
    assertTrue(s.querySubterm(p).equals(constantTerm("d", baseType("b"))));
  }

  @Test
  public void testPartialSubtermGood() {
    Term s = twoArgFuncTerm();
    Position p = new FinalPos(1);
    assertTrue(s.querySubterm(p).toString().equals("f(c)"));
    p = new ArgumentPos(2, new FinalPos(1));
    assertTrue(s.querySubterm(p).toString().equals("g"));
  }

  @Test
  public void testSubtermInHead() {
    // (λx.f(x))(a)
    Variable x = new Binder("x", baseType("o"));
    Term s = new Application(new Abstraction(x, unaryTerm("f", baseType("o"), x)),
                             constantTerm("a", baseType("o")));
    assertTrue(s.querySubterm(new LambdaPos(new ArgumentPos(1, Position.empty))) == x);
  }

  @Test
  public void testSubtermBad() {
    Term s = twoArgVarTerm();
    Position pos = new ArgumentPos(1, new ArgumentPos(2, Position.empty));
    assertThrows(InvalidPositionException.class, () -> s.querySubterm(pos));
  }

  @Test
  public void testHeadSubtermBad() {
    Term s = twoArgFuncTerm();
    Position pos = new ArgumentPos(2, new FinalPos(2));
    assertThrows(InvalidPositionException.class, () -> s.querySubterm(pos));
  }

  @Test
  public void testSubtermReplacementGood() {
    Variable z = new Var("Z", arrowType("Int", "Int"));
    Term s = new Application(z, constantTerm("37", baseType("Int")));
    Term t = s.replaceSubterm(new ArgumentPos(1, Position.empty), s);
    assertTrue(s.toString().equals("Z(37)"));
    assertTrue(t.queryArgument(1).equals(s));
    assertTrue(t.toString().equals("Z(Z(37))"));
  }

  @Test
  public void testSubtermInHeadReplacementGood() throws PositionFormatException {
    // (λxy.f(y,x))(a, b)
    Variable x = new Binder("x", baseType("o"));
    Variable y = new Binder("y", baseType("o"));
    Term f = constantTerm("f", arrowType(baseType("o"), arrowType("o", "o")));
    Term a = constantTerm("a", baseType("o"));
    Term b = constantTerm("b", baseType("o"));
    Term s = new Application(new Abstraction(x, new Abstraction(y, new Application(f, y, x))),
                             a, b);
    
    // replace y in f(y,x) by x
    Term t = s.replaceSubterm(Position.parse("0.0.1"), x);
    assertTrue(t.toString().equals("(λx.λy.f(x, x))(a, b)"));

    // replace f(y) by (λy.y)
    Term u = s.replaceSubterm(Position.parse("0.0.*1"), new Abstraction(y, y));
    assertTrue(u.toString().equals("(λx.λy.(λy1.y1)(x))(a, b)"));
  }

  @Test
  public void testSubtermHeadReplacementGood() {
    Term s = twoArgFuncTerm();  // f(c, g(d))
    Position pos = new ArgumentPos(2, new FinalPos(1));
    Term a = constantTerm("a", baseType("A"));
    Variable x = new Binder("x", arrowType(baseType("A"), arrowType("b", "b")));
    Term t = s.replaceSubterm(pos, x.apply(a));
    assertTrue(t.toString().equals("f(c, x(a, d))"));
    Term q = t.replaceSubterm(pos, x.apply(a));
    assertTrue(t.equals(q));
    pos = new ArgumentPos(2, Position.empty);
    t = s.replaceSubterm(pos, constantTerm("B", baseType("b")));
    assertTrue(t.toString().equals("f(c, B)"));
    assertTrue(s.toString().equals("f(c, g(d))"));
  }

  @Test
  public void testSubtermReplacementBadPosition() {
    Variable z = new Var("Z", arrowType("Int", "Int"));
    Term s = new Application(z, constantTerm("37", baseType("Int")));
    assertThrows(InvalidPositionException.class, () ->
      s.replaceSubterm(new ArgumentPos(2, Position.empty), s));
  }

  @Test
  public void testSubtermHeadReplacementBadPosition() {
    Variable z = new Var("Z", arrowType("Int", "Int"));
    Term s = new Application(z, constantTerm("37", baseType("Int")));
    assertThrows(InvalidPositionException.class, () ->
      s.replaceSubterm(new ArgumentPos(2, new FinalPos(1)), s));
  }

  @Test
  public void testSubtermHeadReplacementBadHeadPosition() {
    Constant f = new Constant("f", arrowType("Int", "Int"));
    Term s = new Application(f, constantTerm("37", baseType("Int")));
    assertThrows(InvalidPositionException.class, () ->
      s.replaceSubterm(new FinalPos(2), constantTerm("a", baseType("B"))));
  }

  @Test
  public void testSubtermReplacementBadTypeSub() {
    Constant f = new Constant("f", arrowType("Int", "Bool"));
    Term s = new Application(f, constantTerm("37", baseType("Int")));
    assertThrows(TypingException.class, () ->
      s.replaceSubterm(new ArgumentPos(1, Position.empty), s));
  }

  @Test
  public void testSubtermReplacementBadTypeTop() {
    Variable z = new Binder("Z", arrowType("Int", "Bool"));
    Term s = new Application(z, constantTerm("37", baseType("Int")));
    assertThrows(TypingException.class, () ->
      s.replaceSubterm(Position.empty, constantTerm("42", baseType("Int"))));
  }

  @Test
  public void testSubtermHeadReplacementBadType() {
    Variable z = new Binder("Z", arrowType("Int", "Bool"));
    Term s = new Application(z, constantTerm("37", baseType("Int")));
    Term replacement = constantTerm("f", arrowType("Int", "Int"));
    assertThrows(TypingException.class, () ->
      s.replaceSubterm(new FinalPos(1), replacement));
  }

  @Test
  public void testApplyingBaseTerm() {
    Term s = twoArgVarTerm();
    Term t = constantTerm("37", baseType("Int"));
    assertThrows(TypingException.class, () -> s.apply(t));
  }

  @Test
  public void testApplyingBadType() {
    Type o = baseType("o");
    Type a = baseType("a");
    Type type = arrowType(a, arrowType(o, a));
    Term c = constantTerm("c", a);
    FunctionSymbol f = new Constant("f", type);
    Term fc = new Application(f, c);
    assertThrows(TypingException.class, () -> fc.apply(c));
  }

  @Test
  public void testCorrectApplication() {
    Type o = baseType("o");
    Type a = baseType("a");
    Type type = arrowType(a, arrowType(o, a));
    Term c = constantTerm("c", arrowType(a, a));
    Term d = constantTerm("d", a);
    Variable x = new Var("x", type);
    Term xc = new Application(x, c.apply(d));
    Term xcb = xc.apply(constantTerm("b", o));
    assertTrue(xcb.toString().equals("x(c(d), b)"));
  }

  @Test
  public void testRefreshBinders() {
    // (λxy.f(x,y))(g(λz.z), g(λz.z))
    Variable x = new Binder("x", baseType("o"));
    Variable y = new Binder("x", baseType("o"));
    Variable z = new Binder("x", baseType("o"));
    Term f = constantTerm("f", arrowType(baseType("o"), arrowType("o", "o")));
    Term g = constantTerm("g", arrowType(arrowType("o", "o"), baseType("o")));
    Term zz = new Abstraction(z, z);
    Term abs = new Abstraction(x, new Abstraction(y, new Application(f, x, y)));
    Term term = new Application(abs, new Application(g, zz), new Application(g, zz));

    TreeMap<Variable,Variable> map = new TreeMap<Variable,Variable>();
    map.put(y, new Var("y", baseType("o")));
    Term t = term.renameAndRefreshBinders(map);
    assertTrue(t.equals(term));
    Variable a = t.queryVariable();
    Variable b = t.queryHead().queryAbstractionSubterm().queryVariable();
    Variable c = t.queryArgument(1).queryArgument(1).queryVariable();
    Variable d = t.queryArgument(2).queryArgument(1).queryVariable();

    assertTrue(a.compareTo(z) > 0);
    assertTrue(b.compareTo(z) > 0);
    assertTrue(c.compareTo(z) > 0);
    assertTrue(d.compareTo(z) > 0);
    assertFalse(a.equals(b));
    assertFalse(a.equals(c));
    assertFalse(a.equals(d));
    assertFalse(b.equals(c));
    assertFalse(b.equals(d));
    assertFalse(c.equals(d));
  }

  @Test
  public void testWellbehaved() {
    // f(x, λy.g(y, x), λx.x, λz.z, y)
    Variable x = new Binder("x", baseType("a"));
    Variable y = new Binder("y", arrowType("b", "b"));
    Variable z = new Binder("z", baseType("c"));
    Term g = new Constant("g", arrowType(arrowType("b", "b"), arrowType("a", "d")));
    Term f = new Constant("f", arrowType(
      baseType("a"), arrowType(
      arrowType(arrowType("b", "b"), baseType("d")), arrowType(
      arrowType("a", "a"), arrowType(
      arrowType("c", "c"), arrowType(
      arrowType("b", "b"), baseType("e")))))));
    Term arg2 = new Abstraction(y, new Application(g, y, x));
    Term arg3 = new Abstraction(x, x);
    Term arg4 = new Abstraction(z, z);
    Term s = new Application(new Application(new Application(f, x, arg2), arg3, arg4), y);
    assertTrue(s.queryArgument(1) == x);
    assertTrue(s.queryArgument(2).queryVariable() != y);
    assertTrue(s.queryArgument(2).queryAbstractionSubterm().queryArgument(1) ==
               s.queryArgument(2).queryVariable());
    assertTrue(s.queryArgument(2).queryAbstractionSubterm().queryArgument(2) == x);
    assertTrue(s.queryArgument(3).queryVariable() != x);
    assertTrue(s.queryArgument(3).queryAbstractionSubterm().queryVariable() ==
               s.queryArgument(3).queryVariable());
    assertTrue(s.queryArgument(4).queryVariable() == z);
    assertTrue(s.queryArgument(5).queryVariable() == y);
  }

  @Test
  public void testVarTermEquality() {
    Variable x = new Binder("x", baseType("o"));
    Variable y = new Var("y", arrowType(baseType("o"), arrowType("o", "o")));
    Variable x2 = new Binder("x2", baseType("o"));
    List<Term> empty = new ArrayList<Term>();
    Term s1 = x;
    Term s2 = y;
    Term s3 = new Application(y, x);
    Term s4 = new Application(y, x, x);
    Term s5 = new Application(y, x, x);
    Term s6 = new Application(y, x, x2);
    
    assertTrue(s1.equals(s1));
    assertFalse(s1.equals(s2));
    assertFalse(s2.equals(s3));
    assertFalse(s3.equals(s4));
    assertFalse(s4.equals(s3));
    assertTrue(s4.equals(s5));
    assertFalse(s5.equals(s6));

    assertTrue(s4.hashCode() == s5.hashCode());
    TreeMap<Variable,Integer> mu = new TreeMap<Variable,Integer>();
    mu.put(x, 3);
    assertTrue(s5.hashCode(mu) != s6.hashCode(mu));
    mu.put(x2, 3);
    assertTrue(s5.hashCode(mu) == s6.hashCode(mu));
  }

  @Test
  public void testFunctionalTermEquality() {
    Term s1 = constantTerm("x", baseType("o"));
    Term s2 = unaryTerm("x", baseType("o"), constantTerm("y", baseType("a")));
    Term s3 = unaryTerm("x", baseType("o"), constantTerm("y", baseType("a")));
    Term s4 = unaryTerm("x", baseType("a"), constantTerm("y", baseType("a")));
    assertFalse(s1.equals(s2));
    assertFalse(s2.equals(s1));
    assertTrue(s2.equals(s3));
    assertFalse(s2.equals(s4));
    assertFalse(s1.equals(new Var("x", baseType("o"))));
  }

  @Test
  public void testAlphaEquality() {
    // (λx.x) (f(y, λx.x)) =[y:=1,z:=1] (λy.y) (f(z, λx.x))
    Variable x = new Binder("x", baseType("o"));
    Variable y = new Binder("y", baseType("o"));
    Variable z = new Binder("z", baseType("o"));
    Constant f = new Constant("f", arrowType(baseType("o"), arrowType(
      arrowType("o", "o"), baseType("o"))));
    TreeMap<Variable,Integer> mu = new TreeMap<Variable,Integer>();
    TreeMap<Variable,Integer> xi = new TreeMap<Variable,Integer>();
    mu.put(y, 1);
    xi.put(z, 1);
    Term xx = new Abstraction(x, x);
    Term s = new Application(xx, new Application(f, y, xx));
    Term t = new Application(new Abstraction(y, y), new Application(f, z, xx));
    assertTrue(s.equals(s));
    assertFalse(s.equals(t));
    assertTrue(s.alphaEquals(t, mu, xi, 2));
  }

  @Test
  public void testFreeVariableRenaming() {
    Variable a = new Binder("x", arrowType(baseType("o"), arrowType("o", "o")));
    Variable b = new Var("x", baseType("o"));
    Variable c = new Var("x", baseType("o"));
    Term combi = new Application(a, b, c);
    assertTrue(combi.toString().equals("x__3(x__1, x__2)"));
  }

  @Test
  public void testCorrectPrintingWithoundVariables() {
    // (λx0.x0)(f(g(x1, x2), λx3.g(x2, x3), λx4.λx3.g(x3,x4)))
    Variable x0 = new Binder("x", baseType("o"));
    Variable x1 = new Binder("x", baseType("o"));
    Variable x2 = new Binder("x", baseType("o"));
    Variable x3 = new Binder("x", baseType("o"));
    Variable x4 = new Binder("x", baseType("o"));
    Term g = constantTerm("g", arrowType(baseType("o"), arrowType("o", "o")));
    Term f = constantTerm("f", arrowType(baseType("o"), arrowType(
      arrowType("o", "o"), arrowType(arrowType(baseType("o"), arrowType("o", "o")),
      baseType("o")))));
    Term abs = new Abstraction(x0,x0);
    Term arg1 = new Application(g, x1, x2);
    Term arg2 = new Abstraction(x3, new Application(g, x2, x3));
    Term arg3 = new Abstraction(x4, new Abstraction(x3, new Application(g, x3, x4)));
    Term total = new Application(abs, (new Application(f, arg1, arg2)).apply(arg3));
    assertTrue(total.toString().equals(
      "(λx1.x1)(f(g(x__1, x__2), λx1.g(x__2, x1), λx1.λx2.g(x2, x1)))"));
  }
}
