/**************************************************************************************************
 Copyright 2024--2025 Cynthia Kop

 Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 in compliance with the License.
 You may obtain a copy of the License at

     http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software distributed under the
 License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 express or implied.
 See the License for the specific language governing permissions and limitations under the License.
 *************************************************************************************************/

package cora.io;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;
import static org.junit.jupiter.api.Assertions.*;

import java.util.List;
import java.util.Set;
import charlie.util.Pair;
import charlie.types.Type;
import charlie.terms.position.Position;
import charlie.terms.replaceable.Renaming;
import charlie.terms.*;
import charlie.trs.*;
import charlie.reader.CoraInputReader;

public class OutputModuleTest {
  private TRS exampleTrs() {
    return CoraInputReader.readTrsFromString("f :: Int -> Int -> Int\na::Int -> Int\n" +
                                             "f(x,y) -> f(y,x) | x> y\n" +
                                             "a(x) -> 3");
  }

  @Test
  public void testPrintTwoParagraphs() {
    OutputModule o = OutputModule.createUnicodeModule(exampleTrs());
    o.print("Hello ");
    o.println("world!");
    o.print("Test.");
    o.println();
    assertTrue(o.toString().equals("Hello world!\n\nTest.\n\n"));
  }

  @Test
  public void testPrintEmptyParagraph() {
    OutputModule o = OutputModule.createPlainModule(exampleTrs());
    o.println();
    o.println("Beep!");
    o.println();
    o.println("Boop!");
    assertTrue(o.toString().equals("\nBeep!\n\n\nBoop!\n\n"));
  }

  @Test
  public void testPrintSimpleTable() {
    OutputModule o = OutputModule.createPlainModule(exampleTrs());
    o.startTable();
    o.print("A");
    o.nextColumn();
    o.print("B");
    o.println();
    o.nextColumn("C");
    o.nextColumn("D");
    o.endTable();
    assertTrue(o.toString().equals("  A B\n  C D\n\n"));
  }

  @Test
  public void testPadding() {
    OutputModule o = OutputModule.createPlainModule(exampleTrs());
    o.startTable();
    o.nextColumn("Hello");
    o.nextColumn("a");
    o.println();
    o.nextColumn("bb");
    o.print("Super long");
    o.println(" line!");
    o.endTable();
    assertTrue(o.toString().equals("  Hello a\n  bb    Super long line!\n\n"));
  }

  @Test
  public void testEmptyTable() {
    OutputModule o = OutputModule.createPlainModule(exampleTrs());
    o.startTable();
    o.endTable();
    assertTrue(o.toString().equals("\n"));
  }

  @Test
  public void testAlmostEmptyTable() {
    OutputModule o = OutputModule.createPlainModule(exampleTrs());
    o.startTable();
    o.println();
    o.endTable();
    assertTrue(o.toString().equals("  \n\n"));
  }

  @Test
  public void testTableAndParagraph() {
    OutputModule o = OutputModule.createPlainModule(exampleTrs());
    o.startTable();
    o.nextColumn("ABCD");
    o.println("x");
    o.println();
    o.println("E");
    o.endTable();
    o.println("Some more testing");
    assertTrue(o.toString().equals("  ABCD x\n  \n  E\n\nSome more testing\n\n"));
  }

  @Test
  public void testParagraphAndTable() {
    OutputModule o = OutputModule.createPlainModule(exampleTrs());
    o.println("Hello");
    o.startTable();
    o.println("bing");
    o.endTable();
    o.println("Bye.");
    assertTrue(o.toString().equals("Hello\n\n  bing\n\nBye.\n\n"));
  }

  @Test
  public void testCodes() {
    OutputModule p = OutputModule.createPlainModule(exampleTrs());
    p.println("%{ruleArrow}, %{typeArrow}, %{lambda}, %{vdash}");
    assertTrue(p.toString().equals("->, ->, \\, |-\n\n"));
    OutputModule u = OutputModule.createUnicodeModule(exampleTrs());
    u.println("%{ruleArrow}, %{typeArrow}, %{lambda}, %{vdash}");
    assertTrue(u.toString().equals("→, →, λ, ⊢\n\n"));
  }

  @Test
  public void testTermWithStringCodes() {
    OutputModule o = OutputModule.createPlainModule(exampleTrs());
    assertThrows(IllegalPrintException.class, () ->
      o.println("a %a bing%s %{lambda} %% %a gg!", "hel%alo", "test?"));
    o.println("a %a bing%a %{lambda} %% %a gg!", "hel%alo", "s", "test?");
    assertTrue(o.toString().equals("a hel%alo bings \\ % test? gg!\n\n"));
  }

  @Test
  public void testPrintTermsWithDistinctVariablesWithoutRenaming() {
    TRS trs = exampleTrs();
    OutputModule o = OutputModule.createPlainModule(trs);
    Term a = CoraInputReader.readTerm("f(x, 3)", trs);
    Term b = CoraInputReader.readTerm("f(0, x + y)", trs);
    o.println("First attempt: terms are %a and %a.", a, b);
    o.print("Second attempt: terms are %a", a);
    o.println(" and %a.", b);
    assertTrue(o.toString().equals(
      "First attempt: terms are f(x__1, 3) and f(0, x__2 + y).\n\n" +
      "Second attempt: terms are f(x, 3) and f(0, x + y).\n\n"));
  }

  @Test
  public void testPrintTermsWithRenaming() {
    TRS trs = exampleTrs();
    OutputModule o = OutputModule.createPlainModule(trs);
    Term extra = CoraInputReader.readTerm("f(x, 3)", trs);
    Term a = CoraInputReader.readTerm("f(x, y)", trs);
    Term b = CoraInputReader.readTerm("f(x, y)", trs);
    Renaming renaming = o.generateUniqueNaming(extra, a);
    o.println("First attempt: %a, %a.", a, b);
    o.println("Second attempt: %a, %a.", new Pair<Term,Renaming>(a,renaming), b);
    assertTrue(o.toString().equals("First attempt: f(x__1, y__1), f(x__2, y__2).\n\n" +
                                   "Second attempt: f(x__2, y), f(x, y).\n\n"));
  }

  @Test
  public void testPrintType() {
    TRS trs = exampleTrs();
    Type t = CoraInputReader.readType("(a -> b) -> (c -> d)");
    OutputModule p = OutputModule.createPlainModule(trs);
    OutputModule u = OutputModule.createUnicodeModule(trs);
    p.println("%a", t);
    u.println("%a", t);
    assertTrue(p.toString().equals("(a -> b) -> c -> d\n\n"));
    assertTrue(u.toString().equals("(a → b) → c → d\n\n"));
  }

  @Test
  public void testPrintPosition() throws charlie.terms.position.PositionFormatException {
    OutputModule p = OutputModule.createPlainModule(exampleTrs());
    OutputModule u = OutputModule.createUnicodeModule(exampleTrs());
    Position pos = Position.parse("1.2.☆3");
    p.println("%a", pos);
    u.println("%a", pos);
    assertTrue(p.toString().equals("1.2.*3\n\n"));
    assertTrue(u.toString().equals("1.2.☆3\n\n"));
  }

  @Test
  public void testPrintRule() {
    TRS trs = exampleTrs();
    OutputModule o = OutputModule.createPlainModule(trs);
    Rule r1 = trs.queryRule(0);
    Rule r2 = trs.queryRule(1);
    o.println("Rule %a, rule %a, lhs %a, lhs %a.", r1, r2, r1.queryLeftSide(), r2.queryLeftSide());
    assertTrue(o.toString().equals("Rule f(x, y) -> f(y, x) | x > y, rule a(x) -> 3, " +
                                   "lhs f(x__1, y), lhs a(x__2).\n\n"));
  }

  @Test
  public void testErrorsWhenNotInATable() {
    TRS trs = exampleTrs();
    OutputModule o = OutputModule.createPlainModule(trs);
    o.print("Hello");
    assertThrows(IllegalPrintException.class, () -> o.endTable());
    assertThrows(IllegalPrintException.class, () -> o.nextColumn());
    o.startTable();
    o.print("Ping");
    o.startTable();
    o.print("Pong");
    o.endTable();
    assertTrue(o.toString().equals("Hello\n\n  Ping\n\n  Pong\n\n"));
  }

  @Test
  public void testIncorrectNumberOfArguments() {
    TRS trs = exampleTrs();
    OutputModule o = OutputModule.createPlainModule(trs);
    assertThrows(IllegalPrintException.class, () -> o.println("Test 1: %a, %a, %a", "a", "b"));
    assertThrows(IllegalPrintException.class, () -> o.print("Test 2: %a, %a", "a", "b", "c"));
  }

  @Test
  public void testPrintTrs() {
    TRS trs = exampleTrs();
    OutputModule o = OutputModule.createUnicodeModule(trs);
    o.printTrs(trs);
    assertTrue(o.toString().equals(
      "Cora-TRS with rule schemes Beta and Calc:\n\n" +
      "  Signature: a :: Int → Int\n" +
      "             f :: Int → Int → Int\n\n" +
      "  Rules: f(x, y) → f(y, x) | x > y\n" +
      "         a(x) → 3\n\n"));
  }

  @Test
  public void testPrintEmptyTrs() {
    TRS trs = TrsFactory.createTrs(new Alphabet(List.of()), List.of(), TrsFactory.AMS);
    OutputModule o = OutputModule.createUnicodeModule(trs);
    o.printTrs(trs);
    assertTrue(o.toString().equals(
      "AMS with only rule scheme Beta:\n\n" +
      "  Signature: (empty)\n\n" +
      "  Rules: (empty)\n\n"));
  }
}

