package jason.client;

import com.sun.javacard.javax.smartcard.rmiclient.CardAccessor;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.net.MalformedURLException;
import java.rmi.*;
import java.rmi.server.*;
import java.security.*;
import java.util.Arrays;
import javax.crypto.*;
import jason.Constants;
import jason.terminal.RemoteCardAccessor;

public class SecureCardAccessor implements CardAccessor, Constants {
	public static final byte ROLE_CARD = (byte) 0;

	private RemoteCardAccessor remoteCardAccessor;
	private short id;
	private byte[] jdf;
	private byte role;
	private MyCipher cipher;
	private SecureRandom secureRandom;
	private MySignature signature;
	private KeyStore keyStore;
	private XORKey sessionKey;

	private static byte[] loginData = {(byte) 0x80, (byte) 0x39, (byte) 0x02, (byte) 0x02};

	/**
	 * Initialises all values to the default values.
	 */
	public SecureCardAccessor() {
		remoteCardAccessor = null;
		id = (short) 0;
		cipher = null;
		signature = null;
		secureRandom = null;
	}

	/**
	 * Calls {@link #selectAPDU} of {@link #invokeAPDU} according to the command.
	 * @param sendData Eiter a select APDU command or an invoke APDU command
	 * @return The response APDU
	 * @throws IOException when an exception is thrown by {@link #selectAPDU} or {@link #invokeAPDU}
	 */
	public byte[] exchangeAPDU(byte[] sendData) throws IOException {
		System.out.println("Received from client: " + arrayToString(sendData));
		byte[] receiveData;
		if (sendData[0] == 0x00 && sendData[1] == (byte) 0xA4)
			receiveData = selectAPDU(sendData);
		else
			receiveData = invokeAPDU(sendData);
		System.out.println("Send to client:       " + arrayToString(receiveData));
		return receiveData;
	}

	/**
	 * Returns the session identifier. The session identifier equals the object id
	 * of the remote object.
	 * @return Session identifier
	 */
	public short getSessionIdentifier() {
		return id;
	}

	/**
	 * Connects to a CardAccessor run on another host.
	 * @param host The foreign host
	 * @param port The foreign port number
	 * @throws RemoteException when the connection cannot be set up
	 */
	public void setRemote(String host, int port) throws RemoteException {
		try {
			remoteCardAccessor = (RemoteCardAccessor) Naming.lookup("//" + host + ":" + port + "/RemoteCardAccessor");
		}
		catch (MalformedURLException mue) {
			throw new RemoteException(mue.getMessage());
		}
		catch (NotBoundException nbe) {
			throw new RemoteException(nbe.getMessage());
		}
	}

	/**
	 * Logs in with the specified role byte given the keystore.
	 * @param role role byte
	 * @param keyStore key store containing a key for the role
	 * @return true if allowed to log in, false otherwise
	 * @throws IOException when something went wrong
	 */
	public boolean login(byte role, KeyStore keyStore) throws IOException {
		this.keyStore = keyStore;
		this.role = role;
		System.out.println("Logging in as role " + role + "...");
		try {
			XORPrivateKey key = (XORPrivateKey) keyStore.getKey(role);
			System.out.println("role private key:     " + arrayToString(key.getEncoded()));
			System.out.println("role public key:      " + arrayToString(new XORPublicKey(key).getEncoded()));
			if (key == null)
				return false;
			if (secureRandom == null) {
				System.out.println("Initializing SecureRandom...");
				secureRandom = new SecureRandom();
			}

			//Send role, clientRandom.length, clientRandom
			byte[] clientRandom = new byte[8];
			secureRandom.nextBytes(clientRandom);
			byte[] request = new byte[loginData.length + 4 + clientRandom.length];
			System.arraycopy(loginData, 0, request, 0, loginData.length);
			int offset = loginData.length;
			request[offset++] = (byte) (clientRandom.length + 2);
			request[offset++] = role;
			request[offset++] = (byte) clientRandom.length;
			System.arraycopy(clientRandom, 0, request, offset, clientRandom.length);
			offset += clientRandom.length;
			request[offset] = (byte) 0x7F;

			//Receive serverRandom.length, serverRandom, encryptedClientRandom.length, encryptedClientRandom
			System.out.println("Send to client:       " + arrayToString(request));
			byte[] response = remoteCardAccessor.exchangeAPDU(request);
			System.out.println("Received from client: " + arrayToString(response));
			offset = 2;
			int serverRandomLength = response[offset++];
			byte[] serverRandom = new byte[serverRandomLength];
			System.arraycopy(response, offset, serverRandom, 0, serverRandomLength);
			offset += serverRandomLength;
			int encryptedClientRandomLength = response[offset++];
			XORPublicKey cardPublicKey = (XORPublicKey) keyStore.getKey(ROLE_CARD);
			System.out.println("card public key:      " + arrayToString(cardPublicKey.getEncoded()));
			signature = new MySignature(new XORSignature());
			signature.initVerify(cardPublicKey);
			signature.update(response, offset, encryptedClientRandomLength);
			if (signature.verify(clientRandom)) {
				signature.initSign(key);
				signature.update(serverRandom, 0, serverRandom.length);
				byte[] encryptedServerRandom = signature.sign();
				request = new byte[7 + encryptedServerRandom.length];
				offset = 0;
				request[offset++] = (byte) 0x80; //CLA
				request[offset++] = (byte) 0x39; //INS
				request[offset++] = (byte) 0x02; //Major version number
				request[offset++] = (byte) 0x02; //Minor version number
				request[offset++] = (byte) (encryptedServerRandom.length+1);
				request[offset++] = (byte) encryptedServerRandom.length;
				System.arraycopy(encryptedServerRandom, 0, request, offset, encryptedServerRandom.length);
				offset += encryptedServerRandom.length;
				request[offset++] = (byte) 0x7F; //Expected response length
				System.out.println("Send to client:       " + arrayToString(request));

				//Send encrypted server random and receive session key
				response = remoteCardAccessor.exchangeAPDU(request);
				System.out.println("Received from client: " + arrayToString(response));
				//System.out.println("Decrypted client random: " + arrayToString(decryptedClientRandom));
				System.out.println("Client random:           " + arrayToString(clientRandom));
				System.out.println("Server random:           " + arrayToString(serverRandom));
				System.out.println("Encrypted server random: " + arrayToString(encryptedServerRandom));
				if (response[0] == (byte) 0x90 &&
						response[1] == (byte) 0x00 &&
						response[2] == role &&
						response[3] == (byte) 1) {
					cipher = new MyCipher(new XORCipher(), new XORProvider(), "XOR");
					cipher.init(Cipher.DECRYPT_MODE, new XORPublicKey(key));
					byte[] sessionKeyData = cipher.doFinal(response, 5, response[4]);
					sessionKey = new XORKey(sessionKeyData);
					cipher.init(Cipher.ENCRYPT_MODE, sessionKey);
					System.out.println("Session key:           " + arrayToString(sessionKeyData));
					return true;
				} else
					return false;
			} else {
				byte[] signedClientRandom = new byte[encryptedClientRandomLength];
				System.arraycopy(response, offset, signedClientRandom, 0, encryptedClientRandomLength);
				System.out.println("Invalid signature");
				throw new InvalidSignatureException("Invalid signature");
			}
		}
		catch (InvalidKeyException ike) { ike.printStackTrace(); }
		catch (IllegalBlockSizeException ibse) { ibse.printStackTrace(); }
		catch (BadPaddingException bpe) { bpe.printStackTrace(); }
		catch (SignatureException se) { se.printStackTrace(); }
		remoteCardAccessor.exchangeAPDU(new byte[] {(byte) 0x80, (byte) 0x39, (byte) 0x02, (byte) 0x02, (byte) 0x00, (byte) 0x7F});
		return false;
	}

	/**
	 * Uploads a key to the key store object on the smart card. Keys can only be
	 * uploaded once. Subsequent uploads will result in a failure
	 * @param role role byte
	 * @param keyType value from KeyBuilder.TYPE_...
	 * @param key the key itself
	 * @param sessionAlgorithm value from Cipher.ALG_...
	 * @return true if the key could be uploaded, false otherwise
	 * @throws IOException when something else went wrong
	 */
	public boolean putKey(byte role, byte keyType, Key key, byte sessionAlgorithm) throws IOException {
		byte[] keyData = key.getEncoded();
		byte[] request = new byte[keyData.length + 10];
		int offset = 0;
		request[offset++] = (byte) 0x80; //CLA
		request[offset++] = (byte) 0x40; //INS_PUT_KEY
		request[offset++] = (byte) 0x02; //Major version number
		request[offset++] = (byte) 0x02; //Minor version number
		request[offset++] = (byte) (4+keyData.length);
		request[offset++] = role;
		request[offset++] = sessionAlgorithm;
		request[offset++] = keyType;
		request[offset++] = (byte) (keyData.length);
		System.arraycopy(keyData, 0, request, offset, keyData.length);
		offset += keyData.length;
		request[offset++] = 0x7F;
		byte[] response = remoteCardAccessor.exchangeAPDU(request);
		return response != null && response.length >= 2 && response[0] == (byte) 0x90 && response[1] == 0x00;
	}

	/**
	 * Select the card applet by its AID. The card response is parsed and the name
	 * of the applet is substracted and used to find the stub object. The stub is
	 * used to get the JDF array containing the security requirements.
	 * @param sendData Select APDU command
	 * @return response APDU
	 * @throws IOException when something went wrong
	 */
	private byte[] selectAPDU(byte[] sendData) throws IOException {
		byte[] receiveData = remoteCardAccessor.exchangeAPDU(sendData);
		System.out.println("Received from card:   " + arrayToString(receiveData));
		if (receiveData[2] == FCI_TAG &&
				receiveData[4] == APPLICATION_DATA_TAG &&
				receiveData[6] == JC_RMI_DATA_TAG &&
				receiveData[11] == NORMAL_TAG) { //select_response
			id = (short) ((receiveData[12]>>8 | receiveData[13]) & 0xFFFF);
			int offset = 15 + receiveData[14];
			int length = receiveData[offset++];
			String packageName;
			try {
				packageName = new String(receiveData, offset, length, "UTF-8").replace('/', '.');
			}
			catch (UnsupportedEncodingException e1) { packageName = ""; }
			offset += length;
			length = receiveData[offset++];
			String className;
			try {
				className = new String(receiveData, offset, length, "UTF-8");
			}
			catch (UnsupportedEncodingException e2) { className = ""; }
			String fullClassName = packageName + "." + className;
			System.out.println("Select " + fullClassName);
			offset += length;

			//Get Stub object
			try {
				Class remoteClass = Class.forName(fullClassName + "_Stub");
				Constructor remoteConstructor = remoteClass.getConstructor(new Class[] {RemoteRef.class});
				Stub remoteStub = (Stub) remoteConstructor.newInstance(new Object[] {null});
				jdf = remoteStub.getJDF();
				System.out.println("jdf = " + arrayToString(jdf));
			}
			catch (ClassNotFoundException cnfe) { jdf = null; }
			catch (NoSuchMethodException nsme) { jdf = null; }
			catch (InstantiationException ie) { jdf = null; }
			catch (IllegalAccessException iae) { jdf = null; }
			catch (InvocationTargetException ite) { jdf = null; }
		}
		return receiveData;
	}

	/**
	 * Encrypts and signs the parameters when necessary and decrypt / verifies
	 * the response from the card
	 * @param sendData The marshalled parameters in plain text
	 * @return The marshalled result in plain text
	 * @throws JasonSecurityException when the card returns an ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED
	 * @throws InvalidSignatureException when the card signature does not match the locally calculated signature
	 * @throws InvalidFreshnessCounterException when the authentic response does not carry the correct freshness counter
	 */
	private byte[] invokeAPDU(byte[] sendData) throws IOException {
		int jdfOffset = 0;
		int numberOfMethods = jdf[jdfOffset++];
		for (int i=0; i<numberOfMethods; i++)
			if (jdf[jdfOffset] != sendData[7] || jdf[jdfOffset+1] != sendData[8]) { //Incorrect method id
				jdfOffset += 2; //skip method_id
				jdfOffset += jdf[jdfOffset] + 2; //skip number_of_roles, roles and modifier
				jdfOffset += jdf[jdfOffset] + 1; //skip number_of_parameters and parameters
			} else {
				jdfOffset += 2; //skip method_id
				byte numberOfRoles = jdf[jdfOffset++];
				boolean loggedIn = numberOfRoles == 0;
				boolean signaturePresent = numberOfRoles != 0;
				for (int j=0; j<numberOfRoles; j++) {
					loggedIn |= jdf[jdfOffset] == role | jdf[jdfOffset] == ANYBODY;
					signaturePresent &= jdf[jdfOffset++] != ANYBODY;
				}
				if (!loggedIn)
					throw new LoginException("You are not correctly logged in");
				byte methodModifier = jdf[jdfOffset++];
				int numberOfParameters = jdf[jdfOffset++]; //skip number_of_parameters
				Appender plain = new Appender(false);
				Appender authentic = new Appender(false);
//        cipher.init(Cipher.ENCRYPT_MODE, sessionKey);
//        signature.initSign(keyStore.getKey(role));
				CipherAppender confidential = new CipherAppender(cipher, false);
				SignatureAppender sign = new SignatureAppender(signature, false);
				signaturePresent |= (methodModifier & SECURITY_AUTHENTIC) == SECURITY_AUTHENTIC;
				int dataOffset = 9; //skip object_id and method_id
				byte freshnessCounter = 0;
				for (int j=0; j<numberOfParameters; j++)
					signaturePresent |= (jdf[jdfOffset+j] & SECURITY_AUTHENTIC) == SECURITY_AUTHENTIC;
				if (signaturePresent) {
					sign.append(sendData, 7, 2); //method_id
					keyStore.increaseFreshnessCounter(role);
					freshnessCounter = keyStore.getFreshnessCounter(role);
					sign.append(freshnessCounter);
				}
				for (int j=0; j<numberOfParameters; j++) {
					int dataLength = 0;
					switch (jdf[jdfOffset+j] & 0x07) {
						case TYPE_BYTE:
						case TYPE_BOOLEAN: dataLength = 1; break;
						case TYPE_SHORT: dataLength = 2; break;
						case TYPE_INT: dataLength = 4; break;
						default: dataLength = 0;
					}
					if ((jdf[jdfOffset+j] & TYPE_ARRAY) == TYPE_ARRAY)
						dataLength = dataLength*sendData[dataOffset] + 1;
					switch (jdf[jdfOffset+j] & 0xF0) {
						case SECURITY_PLAIN:
							plain.append(sendData, dataOffset, dataLength);
							break;
						case SECURITY_CONFIDENTIAL_AUTHENTIC:
							sign.append(sendData, dataOffset, dataLength);
						case SECURITY_CONFIDENTIAL:
							confidential.append(sendData, dataOffset, dataLength);
							break;
						case SECURITY_AUTHENTIC:
							authentic.append(sendData, dataOffset, dataLength);
							sign.append(sendData, dataOffset, dataLength);
							break;
					}
					dataOffset += dataLength;
				}

				byte[] plainBuffer = plain.getBuffer();
				byte[] confidentialBuffer = confidential.getBuffer();
				byte[] authenticBuffer = authentic.getBuffer();
				byte[] signBuffer = sign.getBuffer();
				Appender all = new Appender(false);
				all.append(sendData, 0, 9); //object_id and method_id
				all.append((byte) plainBuffer.length);
				all.append(plainBuffer);
				all.append((byte) confidentialBuffer.length);
				all.append(confidentialBuffer);
				all.append((byte) authenticBuffer.length);
				all.append(authenticBuffer);
				all.append(freshnessCounter);
				if (signaturePresent) {
					all.append((byte) signBuffer.length);
					all.append(signBuffer);
				} else
					all.append((byte) 0);
				all.append((byte) 0x7F);
				byte[] encryptedBuffer = all.getBuffer();
				encryptedBuffer[4] = (byte) (9+plainBuffer.length+confidentialBuffer.length+authenticBuffer.length);
				if (signaturePresent)
					encryptedBuffer[4] += signBuffer.length;
				System.out.println("Send to card:         " + arrayToString(encryptedBuffer));

				//Send encrypted data and receive encrypted response
				byte[] response = null;
				try {
					byte[] decryptedResponse = null;
					byte[] encryptedResponse = remoteCardAccessor.exchangeAPDU(encryptedBuffer);
					System.out.println("Received from card:   " + arrayToString(encryptedResponse));
					if (encryptedResponse[0] == (byte) 0x69 && encryptedResponse[1] == (byte) 0x82)
						throw new JasonSecurityException("For security reasons the card does not accept the command");
					if (encryptedResponse[0] != (byte) 0x90 || encryptedResponse[1] != 0x00)
						throw new RemoteException("Card error, SW = 0x" + Integer.toHexString(((encryptedResponse[0]&0xFF) << 8 | (encryptedResponse[1]&0xFF))));
					if (encryptedResponse[2] != (byte) 0x81) //Card Exception or card error
						return encryptedResponse;
					int dataLength;
					switch ((byte) (methodModifier & 0x07)) {
						case TYPE_BYTE:
						case TYPE_BOOLEAN: dataLength = 1; break;
						case TYPE_SHORT: dataLength = 2; break;
						case TYPE_INT: dataLength = 4; break;
						default: dataLength = 0;
					}
					boolean isArray = (methodModifier&TYPE_ARRAY) == TYPE_ARRAY;
					switch ((byte) (methodModifier & 0xF0)) {
						case SECURITY_PLAIN:
							if (isArray)
								dataLength = dataLength*encryptedResponse[3] + 1;
							decryptedResponse = new byte[dataLength];
							System.arraycopy(encryptedResponse, 3, decryptedResponse, 0, dataLength);
							break;
						case SECURITY_CONFIDENTIAL:
							cipher.init(Cipher.DECRYPT_MODE, sessionKey);
							decryptedResponse = cipher.doFinal(encryptedResponse, 3, encryptedResponse.length-3);
							break;
						case SECURITY_CONFIDENTIAL_AUTHENTIC:
							cipher.init(Cipher.DECRYPT_MODE, sessionKey);
							int confidentialLength = encryptedResponse[3];
							int signatureLength = encryptedResponse.length-confidentialLength-5;
							decryptedResponse = cipher.doFinal(encryptedResponse, 4, confidentialLength);
							try {
								signature.initVerify((PublicKey) keyStore.getKey(ROLE_CARD));
								freshnessCounter = encryptedResponse[4+confidentialLength];
								checkFreshnessCounter(freshnessCounter);
								signature.update(freshnessCounter);
								signature.update(decryptedResponse, 0, decryptedResponse.length);
								byte[] signData = new byte[signatureLength];
								System.arraycopy(encryptedResponse, 5+confidentialLength, signData, 0, signatureLength);
								if (!signature.verify(signData))
									throw new InvalidSignatureException("Signature of response data is invalid");
							}
							catch (SignatureException se) {
								System.out.println("Signature could not be checked");
								se.printStackTrace();
							}
							break;
						case SECURITY_AUTHENTIC:
							if (isArray)
								dataLength = dataLength*encryptedResponse[3] + 1;
							decryptedResponse = new byte[dataLength];
							System.arraycopy(encryptedResponse, 3, decryptedResponse, 0, dataLength);
							try {
								signature.initVerify((PublicKey) keyStore.getKey(ROLE_CARD));
								freshnessCounter = encryptedResponse[3+dataLength];
								checkFreshnessCounter(freshnessCounter);
								signature.update(freshnessCounter);
								signature.update(decryptedResponse, 0, dataLength);
								byte[] signData = new byte[encryptedResponse.length-4-dataLength];
								System.arraycopy(encryptedResponse, 4+dataLength, signData, 0, signData.length);
								if (!signature.verify(signData))
									throw new InvalidSignatureException("Signature of response data is invalid");
							}
							catch (SignatureException se) {
								throw new InvalidSignatureException("Signature could not be checked");
							}
					}
					response = new byte[3+decryptedResponse.length];
					System.arraycopy(encryptedResponse, 0, response, 0, 3);
					System.arraycopy(decryptedResponse, 0, response, 3, decryptedResponse.length);
				}
				catch (BadPaddingException bpe) { bpe.printStackTrace(); }
				catch (IllegalBlockSizeException ibse) { ibse.printStackTrace(); }
				catch (InvalidKeyException ike) { ike.printStackTrace(); }
				return response;
			}
		System.out.println("Method not found");
		return null;
	}

	/**
	 * Checks if the <code>freshnessCounter</code> is the successor of an earlier
	 * freshness counter. If this is true the freshness counter is increased.
	 * @param freshnessCounter The freshnessCounter found in the card response
	 * @throws InvalidFreshnessCounterException when the counter is not a successor of the last counter
	 */
	private void checkFreshnessCounter(byte freshnessCounter) throws InvalidFreshnessCounterException {
		if (keyStore.getFreshnessCounter(role)+1 == freshnessCounter)
			keyStore.increaseFreshnessCounter(role);
		else
			throw new InvalidFreshnessCounterException("FreshnessCounter " + freshnessCounter + " found, but " + (keyStore.getFreshnessCounter(role)+1) + " is expected.");
	}

	/**
	 * Translates a byte array to a string
	 * @param data
	 * @return translation
	 */
	private static String arrayToString(byte[] data) {
		return arrayToString(data, 0, data.length);
	}

	/**
	 * Translates part of a byte array to a string
	 * @param data
	 * @param offset
	 * @param length
	 * @return translation
	 */
	private static String arrayToString(byte[] data, int offset, int length) {
		StringBuffer buffer = new StringBuffer();
		if (data != null)
			for (int i=offset; i<offset+length; i++) {
				String str = Integer.toHexString(data[i] & 0xFF);
				if (str.length() < 2)
					buffer.append('0');
				buffer.append(str);
				buffer.append(' ');
			}
		return buffer.toString();
	}
}