package jason.compiler;

import jason.Constants;
import java.io.*;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.ParseException;
import java.util.ListResourceBundle;

public class Main implements Constants {
	File jasonFile;
	File packageDir;
	File cardDir;
	File pcDir;
	File interfaceFile;
	File stubFile;
	File skeletonFile;
	Parser parser;

	public void makeInterface() throws IOException {
		//Get compiler variables
		String[] roles = parser.getRoles();
		Method[] methods = parser.getMethods();

		//Create output file
		PrintWriter output = new PrintWriter(new FileWriter(interfaceFile));

		//Write package information
		String packageName = parser.getPackageName();
		if (packageName != null) {
			output.println("package " + packageName + ";");
			output.println();
		}

		//Write imports
		String[] imports = parser.getImports();
		if (imports != null) {
			for (int i=0; i<imports.length; i++)
				output.println("import " + imports[i] + ";");
		}
		output.println("import java.rmi.Remote;");
		if (methods.length > 0)
			output.println("import java.rmi.RemoteException;");
		output.println();

		//Write interface information
		output.println("public interface " + parser.getInterfaceName() + " extends Remote {");

		//Write role constants
		output.println("\tpublic static final byte ROLE_ANYBODY = (byte) 0;");
		for (int i=0; i<roles.length; i++)
			output.println("\tpublic static final byte ROLE_" + roles[i].toUpperCase() + " = (byte) " + Integer.toString(i+1) + ";");
		output.println();

		//Write methods
		for (int i=0; i<methods.length; i++) {
			output.print("\tpublic " + methods[i].getType() + " " + methods[i].getName() + "(");
			Parameter[] parameters = methods[i].getParameters();
			boolean first = true;
			for (int j=0; j<parameters.length; j++) {
				if (!first)
					output.print(", ");
				output.print(parameters[j].getType() + " " + parameters[j].getName());
				first = false;
			}
			output.print(")");
			output.print(" throws RemoteException");
			String[] exceptions = methods[i].getExceptions();
			if (exceptions != null && exceptions.length > 0) {
				for (int j=0; j<exceptions.length; j++)
					output.print(", " + exceptions[j]);
			}
			output.println(";");
		}
		output.println("}");
		output.close();
	}

	public void makeStub() throws IOException {
		//Get compiler variables
		Method[] methods = parser.getMethods();

		//Create output file
		sun.rmi.rmic.IndentingWriter output = new sun.rmi.rmic.IndentingWriter(new FileWriter(stubFile), 2);

		//Write comment
		output.pln("// Stub class generated by jason.compiler.Main, do not edit.");
		output.pln("// Contents subject to change without notice.");
		output.pln();

		//Write package information
		String packageName = parser.getPackageName();
		String interfaceName = parser.getInterfaceName();
		if (packageName != null) {
			output.pln("package " + packageName + ";");
			output.pln();
		}

		//Write imports
		output.pln("import jason.client.Stub;");
		output.pln("import java.lang.reflect.Method;");
		output.pln("import java.rmi.Remote;");
		output.pln("import java.rmi.RemoteException;");
		output.pln("import java.rmi.UnexpectedException;");
		output.pln("import java.rmi.server.RemoteRef;");
		output.pln("import java.rmi.server.RemoteStub;");
		String[] imports = parser.getImports();
		if (imports != null) {
			for (int i=0; i<imports.length; i++)
				output.pln("import " + imports[i] + ";");
		}
		output.pln("import java.rmi.Remote;");
		if (methods.length > 0)
			output.pln("import java.rmi.RemoteException;");
		output.pln();

		//Write classname
		output.plnI("public final class " + interfaceName + "Impl_Stub");
		output.pln("extends RemoteStub");
		output.pln("implements " + packageName + "." + interfaceName + ", Remote, Stub {");
		output.pln();

		//Write constants
		output.pln("private static final long serialVersionUID = 2;");
		output.pln();

		//Write method variables
		for (int i=0; i<methods.length; i++)
			output.pln("private static Method $method_" + methods[i].getName() + "_" + i + ";");
		output.pln();

		//Write static instantiation of method variables
		output.plnI("static {");
		output.plnI("try {");
		for (int i=0; i<methods.length; i++) {
			output.p("$method_" + methods[i].getName() + "_" + i + " = " + packageName + "." + interfaceName + ".class.getMethod(\"" + methods[i].getName() + "\", new java.lang.Class[] {");
			Parameter[] parameters = methods[i].getParameters();
			boolean first = true;
			for (int j=0; j<parameters.length; j++) {
				output.p((first ? "" : ", ") + parameters[j].getType() + ".class");
				first = false;
			}
			output.pln("});");
		}
		output.pOln("}");
		output.plnI("catch (NoSuchMethodException e) {");
		output.pln("throw new NoSuchMethodError(\"stub class initialization failed\");");
		output.pOln("}");
		output.pOln("}");
		output.pln();

		//Write constructors
		output.pln("// constructors");
		output.plnI("public " + interfaceName + "Impl_Stub" + "(RemoteRef ref) {");
		output.pln("super(ref);");
		output.pOln("}");
		output.pln();

		//Write method implementations
		for (int i=0; i<methods.length; i++) {
			output.p("// implementation of " + methods[i].getName() + "(");
			Parameter[] parameters = methods[i].getParameters();
			boolean first = true;
			for (int j=0; j<parameters.length; j++) {
				output.p((first ? "" : ", ") + parameters[j].getType());
				first = false;
			}
			output.pln(")");
			output.p("public " + methods[i].getType() + " " + methods[i].getName() + "(");
			first = true;
			for (int j=0; j<parameters.length; j++) {
				output.p((first ? "" : ", ") + parameters[j].getType() + " param" + j);
				first = false;
			}
			output.pln(")");
			output.p("throws RemoteException");
			String[] exceptions = methods[i].getExceptions();
			for (int j=0; j<exceptions.length; j++)
				output.p(", " + exceptions[j]);
			output.plnI(" {");
			output.plnI("try {");
			if (!methods[i].getType().equals("void"))
				output.p("Object result = ");
			output.p("ref.invoke(this, $method_" + methods[i].getName() + "_" + i + ", new Object[] {");
			first = true;
			for (int j=0; j<parameters.length; j++) {
				if (!first)
					output.p(", ");
				else
					first = false;
				String type = parameters[j].getType();
				if (type.equals("byte"))
					output.p("new Byte(param" + j + ")");
				else if (type.equals("boolean"))
					output.p("new Boolean(param" + j + ")");
				else if (type.equals("short"))
					output.p("new Short(param" + j + ")");
				else
					output.p("param" + j);
			}
			output.pln("}, 0L);");
			String type = methods[i].getType();
			if (!type.equals("void"))
				if (type.equals("byte"))
					output.pln("return ((Byte) result).byteValue();");
				else if (type.equals("boolean"))
					output.pln("return ((Boolean) result).booleanValue();");
				else if (type.equals("short"))
					output.pln("return ((Short) result).shortValue();");
				else
					output.pln("return (" + type + ") result;");
			output.pOln("}");
			output.plnI("catch (RuntimeException rte) {");
			output.pln("throw rte;");
			output.pOln("}");
			output.plnI("catch (RemoteException re) {");
			output.pln("throw re;");
			output.pOln("}");
			for (int j=0; j<exceptions.length; j++) {
				output.plnI("catch (" + exceptions[j] + " e" + j + ") {");
				output.pln("throw e" + j + ";");
				output.pOln("}");
			}
			output.plnI("catch (Exception e) {");
			output.pln("throw new UnexpectedException(\"undeclared checked exception\", e);");
			output.pOln("}");
			output.pOln("}");
			output.pln();
		}

		//Write jdf array
		writeJDFArray(output);
		output.plnI("public byte[] getJDF() {");
		output.pln("return JDF;");
		output.pOln("}");

		//Closes stub file
		output.pOln("}");
		output.close();
	}

	public void makeSkeleton() throws IOException {
		//Get compiler variables
		Method[] methods = parser.getMethods();
		String[] roles = parser.getRoles();
		String packageName = parser.getPackageName();
		String interfaceName = parser.getInterfaceName();

		//Create output file
		sun.rmi.rmic.IndentingWriter output = new sun.rmi.rmic.IndentingWriter(new FileWriter(skeletonFile), 2);

		//Write comment
		output.pln("// Skeleton class generated by jason.compiler.Main, do not edit.");
		output.pln("// Contents subject to change without notice.");
		output.pln();

		//Write package information
		if (packageName != null) {
			output.pln("package " + packageName + ";");
			output.pln();
		}

		//Write imports
		output.pln("import jason.server.*;");
		output.pln("import javacard.framework.*;");
		output.pln("import javacard.framework.service.Dispatcher;");
		output.pln("import javacard.framework.service.RMIService;");
		output.pln();

		//Write class definition
		output.plnI("public class " + interfaceName + "_Skel extends Applet {");

		//Write variables
		output.pln("private Dispatcher dispatcher;");

		//Write JDF array
		writeJDFArray(output);

		//Write constructor
		output.plnI("public " + interfaceName + "_Skel() {");
		output.pln("dispatcher = new Dispatcher((short) 4);");
		output.pln(interfaceName + " " + interfaceName.toLowerCase() + " = new " + interfaceName + "Impl();");
		output.pln("Session session = new Session(new KeyStore((short) " + Integer.toString(roles.length+1) + "), JDF);");
		output.pln("dispatcher.addService(session, Dispatcher.PROCESS_INPUT_DATA);");
		output.pln("dispatcher.addService(session, Dispatcher.PROCESS_COMMAND);");
		output.pln("RMIService rmiService = new RMIService(" + interfaceName.toLowerCase() +");");
		output.pln("dispatcher.addService(rmiService, Dispatcher.PROCESS_COMMAND);");
		output.pln("dispatcher.addService(session, Dispatcher.PROCESS_OUTPUT_DATA);");
		output.pln("register();");
		output.pOln("}");
		output.pln();

		//Write install method
		output.plnI("public static void install(byte[] buffer, short offset, byte length) {");
		output.pln("new " + interfaceName + "_Skel();");
		output.pOln("}");
		output.pln();

		//Write select method
		output.plnI("public boolean select() {");
		output.pln("return true;");
		output.pOln("}");
		output.pln();

		//Write process method
		output.plnI("public void process(APDU apdu) throws ISOException {");
		output.pln("dispatcher.process(apdu);");
		output.pln("JCSystem.requestObjectDeletion();");
		output.pOln("}");

		//Close file
		output.pOln("}");
		output.close();
	}

	public void writeJDFArray(sun.rmi.rmic.IndentingWriter output) throws IOException {
		String[] roles = parser.getRoles();
		Method[] methods = parser.getMethods();
		output.p("private static final byte[] JDF = {");
		output.p("(byte) 0x" + Integer.toHexString(methods.length));
		for (int i=0; i<methods.length; i++) {
			short methodID = computeShortHash(methods[i].getName() + getDescriptor(methods[i]));
			output.p(", (byte) 0x" + Integer.toHexString((methodID >> 8) & 0xFF));
			output.p(", (byte) 0x" + Integer.toHexString(methodID & 0xFF));
			String[] accessibleTo = methods[i].getRoles();
			output.p(", (byte) 0x" + Integer.toHexString(accessibleTo.length));
			for (int j=0; j<accessibleTo.length; j++)
				for (int k=0; k<roles.length; k++)
					if (accessibleTo[j].equals(roles[k]))
						output.p(", (byte) 0x" + Integer.toHexString(k+1));
			int modifier = typeToValue(methods[i].getType());
			modifier |= methods[i].isConfidential() ? SECURITY_CONFIDENTIAL : 0;
			modifier |= methods[i].isAuthentic() ? SECURITY_AUTHENTIC : 0;
			output.p(", (byte) 0x" + Integer.toHexString(modifier));
			Parameter[] parameters = methods[i].getParameters();
			output.p(", (byte) 0x" + Integer.toHexString(parameters.length));
			for (int j=0; j<parameters.length; j++) {
				modifier = typeToValue(parameters[j].getType());
				modifier |= parameters[j].isConfidential() ? SECURITY_CONFIDENTIAL : 0;
				modifier |= parameters[j].isAuthentic() ? SECURITY_AUTHENTIC : 0;
				output.p(", (byte) 0x" + Integer.toHexString(modifier));
			}
		}
		output.pln("};");
		output.pln();
	}

	public void makeOldStub() throws IOException {
		//Create output file
		PrintWriter output = new PrintWriter(new FileWriter(stubFile));

		//Write package information
		String packageName = parser.getPackageName();
		if (packageName != null) {
			output.println("package " + packageName + ".terminal;");
			output.println();
		}

		//Write imports
		output.println("import java.io.IOException;");
		output.println("import jason.framework.terminal.Stub;");
		output.println("import jason.security.card.KeyStore;");
		output.println("import " + packageName + ".card." + parser.getInterfaceName() + ";");
		output.println();

		//Write class information
		output.println("public class " + parser.getInterfaceName() + "Stub extends Stub implements " + parser.getInterfaceName() + " {");

		//Write constructors
		output.println("\tpublic " + parser.getInterfaceName() + "Stub(String hostname, int port, KeyStore keyStore) throws IOException {");
		output.println("\t\tconnect(hostname, port, keyStore);");
		output.println("\t}");
		output.println();

		//Write methods
		Method[] methods = parser.getMethods();
		for (int i=0; i<methods.length; i++) {
			output.print("\tpublic " + methods[i].getType() + " " + methods[i].getName() + "(");
			Parameter[] parameters = methods[i].getParameters();
			boolean first = true;
			boolean hasPlain = false;
			boolean hasConfidential = false;
			boolean hasAuthentic = false;
			boolean hasConfidentialAuthentic = false;
			for (int j=0; j<parameters.length; j++) {
				hasPlain |= !parameters[j].isConfidential() && !parameters[j].isAuthentic();
				hasConfidential |= parameters[j].isConfidential() && !parameters[j].isAuthentic();
				hasAuthentic |= parameters[j].isAuthentic() && !parameters[j].isConfidential();
				hasConfidentialAuthentic |= parameters[j].isConfidential() && parameters[j].isAuthentic();
				if (!first)
					output.print(", ");
				output.print(parameters[j].getType() + " " + parameters[j].getName());
				first = false;
			}
			output.println(") {");

			//Check correct role
			String[] roles = methods[i].getRoles();
			boolean containsAnyBody = false;
			for (int j=0; j<roles.length; j++)
				containsAnyBody |= roles[j].equals("ANYBODY");
			if (!containsAnyBody) {
				first = true;
				for (int j=0; j<roles.length; j++) {
					if (first) {
						output.print("\t\tif (");
						first = false;
					} else
						output.print(" && \n\t\t    ");
					output.print("session.getActiveRole() != " + roles[j].toUpperCase());
				}
				output.println(") ");
				first = true;
				for (int j=0; j<roles.length; j++) {
					if (first) {
						output.print("\t\t\tsession.logIn(new byte[] {");
						first = false;
					} else
						output.print(", ");
					output.print(roles[j].toUpperCase());
				}
				output.println("});");
			}

			//Send marshalled parameters
			output.println("\t\tsession.write(" + methods[i].getName().toUpperCase() + ");");
			if (hasPlain) {
				output.println("\t\tsession.setMode(Session.MODE_SEND_PLAIN);");
				for (int j=0; j<parameters.length; j++)
					if (!parameters[j].isConfidential() && !parameters[j].isAuthentic())
						output.println("\t\tsession.write(" + parameters[j].getName() + ");");
			}
			if (hasConfidential) {
				output.println("\t\tsession.setMode(Session.MODE_SEND_CONFIDENTIAL);");
				for (int j=0; j<parameters.length; j++)
					if (parameters[j].isConfidential() && !parameters[j].isAuthentic())
						output.println("\t\tsession.write(" + parameters[j].getName() + ");");
			}
			if (hasConfidentialAuthentic) {
				output.println("\t\tsession.setMode(Session.MODE_SEND_CONFIDENTIAL_AUTHENTIC);");
				for (int j=0; j<parameters.length; j++)
					if (parameters[j].isConfidential() && parameters[j].isAuthentic())
						output.println("\t\tsession.write(" + parameters[j].getName() + ");");
			}
			if (hasAuthentic) {
				output.println("\t\tsession.setMode(Session.MODE_SEND_AUTHENTIC);");
				for (int j=0; j<parameters.length; j++)
					if (!parameters[j].isConfidential() && parameters[j].isAuthentic())
						output.println("\t\tsession.write(" + parameters[j].getName() + ");");
			}

			//Receive result
			if (!methods[i].isConfidential() && !methods[i].isAuthentic())
				output.println("\t\tsession.setMode(Session.MODE_RECEIVE_PLAIN);");
			else if (methods[i].isConfidential() && !methods[i].isAuthentic())
				output.println("\t\tsession.setMode(Session.MODE_RECEIVE_CONFIDENTIAL);");
			else if (methods[i].isConfidential() && methods[i].isAuthentic())
				output.println("\t\tsession.setMode(Session.MODE_RECEIVE_CONFIDENTIAL_AUTHENTIC);");
			else if (!methods[i].isConfidential() && methods[i].isAuthentic())
				output.println("\t\tsession.setMode(Session.MODE_RECEIVE_AUTHENTIC);");
			output.println("\t\tbyte exception=session.readByte();");
			output.println("\t\t//Handel hier de exception af");
			if (methods[i].getType().equals("byte"))
				output.println("\t\treturn session.readByte();");
			else if (methods[i].getType().equals("boolean"))
				output.println("\t\treturn session.readBoolean();");
			else if (methods[i].getType().equals("short"))
				output.println("\t\treturn session.readShort();");
			else if (methods[i].getType().equals("byte[]"))
				output.println("\t\treturn session.readByteArray();");

			/* Dit is oud!
			if (parameters != null && parameters.length != 0) {
				output.println("\t\tAppender plain = new Appender();");
				if (hasConfidential)
					output.println("\t\tCipherAppender confidential = new CipherAppender(session.getCipher());");
				if (hasAuthentic)
					output.println("\t\tAppender authentic = new Appender();");
				if (hasAuthentic || hasConfidentialAuthentic)
					output.println("\t\tSignatureAppender signature = new SignatureAppender(session.getSignature());");
			}
			for (int j=0; j<parameters.length; j++)
				if (!parameters[j].isAuthentic() && !parameters[j].isConfidential())
					output.println("\t\tplain.append(" + parameters[j].getName() + ");");
				else if (parameters[j].isConfidential() && !parameters[j].isAuthentic())
					output.println("\t\tconfidential.append(" + parameters[j].getName() + ");");
				else if (parameters[j].isConfidential() && parameters[j].isAuthentic()) {
					output.println("\t\tconfidential.append(" + parameters[j].getName() + ");");
					output.println("\t\tsignature.append(" + parameters[j].getName() + ");");
				} else {
					output.println("\t\tauthentic.append(" + parameters[j].getName() + ");");
					output.println("\t\tsignature.append(" + parameters[j].getName() + ");");
				}
			if (hasConfidential)
				output.println("\t\tplain.append(confidential);");
			if (hasAuthentic)
				output.println("\t\tplain.append(authentic);");
			if (hasAuthentic || hasConfidentialAuthentic)
				output.println("\t\tplain.append(signature);");
			if (parameters != null && parameters.length != 0) {
				output.println("\t\ttry {");
				output.println("\t\t\t//Send parameters");
				output.println("\t\t\tObjectOutputStream output = new ObjectOutputStream(socket.getOutputStream());");
				output.println("\t\t\toutput.writeInt(plain.getBuffer().length);");
				output.println("\t\t\toutput.write(plain.getBuffer());");
				String[] exceptions = methods[i].getExceptions();
				output.println("\t\t\tObjectInputStream input = new ObjectInputStream(socket.getInputStream());");
				output.println("\t\t\tint responseLength = input.readInt();");
				if (exceptions != null && exceptions.length != 0) {
					output.println("\t\t\tresponseLength--;");
					output.println("\t\t\t//Receive exceptions status");
					output.println("\t\t\tbyte exception = input.readByte();");
					output.println("\t\t\t//throw the correct exceptions");
				}
				if (methods[i].getType().equals("byte"))
					output.println("\t\t\tbyte result = input.readByte();");
				else if (methods[i].getType().equals("short"))
					output.println("\t\t\tshort result = input.readShort();");
				output.println("\t\t}");
				output.println("\t\tcatch (IOException e) {");
				output.println("\t\t\t//Do something");
				output.println("\t\t}");
			}
			*/
			output.println("\t}");
			output.println();
		}

		output.println("}");
		output.close();
	}

	public void makeOldSkeleton() throws IOException {
		//Create output file
		PrintWriter output = new PrintWriter(new FileWriter(skeletonFile));

		//Write package information
		String packageName = parser.getPackageName();
		if (packageName != null) {
			output.println("package " + packageName + ".card;");
			output.println();
		}

		//Write import
		output.println("import jason.framework.card.Gateway;");
		output.println("import jason.security.card.KeyStore;");
		output.println("import jason.security.card.Session;");
		output.println("import javacard.framework.*;");
		output.println();

		//Write class information
		output.println("public class " + parser.getInterfaceName() + "Skeleton extends Applet implements Gateway {");

		output.println("\tprivate byte[] incomingBytes;");
		output.println("\tprivate byte[] outgoingBytes;");
		output.println("\tprivate " + parser.getInterfaceName() + " " + parser.getInterfaceName().toLowerCase() + ";");
		output.println("\tprivate KeyStore keystore;");
		output.println("\tprivate Session session;");
		output.println();

		//Constructor
		output.println("\tprivate " + parser.getInterfaceName() + "Skeleton() {");
		output.println("\t\t" + parser.getInterfaceName().toLowerCase() + " = new " + parser.getInterfaceName() + "Impl();");
		String[] roles = parser.getRoles();
		output.println("\t\tkeystore = new KeyStore((byte) " + roles.length + ");");
		output.println("\t\tsession = new Session(keystore);");
		output.println("\t\tregister();");
		output.println("\t}");
		output.println();

		//Process() method
		Method[] methods = parser.getMethods();
		output.println("\tpublic void process() throws ISOException {");
		output.println("\t\tsession.unmarshall(incomingBytes, (short) 0, (short) (incomingBytes.length));");
		output.println("\t\tswitch (incomingBytes[0]) {");
		for (int i=0; i<methods.length; i++) {
			output.println("\t\t\tcase " + parser.getInterfaceName() + "." + methods[i].getName().toUpperCase() + ": {");

			//Check if correct role
			String[] possibleRoles = methods[i].getRoles();
			boolean containsAnybody = false;
			for (int j=0; j<possibleRoles.length; j++)
				containsAnybody |= possibleRoles[j].equals("ANYBODY");
			if (!containsAnybody) {
				output.print("\t\t\t\tif (session.getActiveRole() != ");
				boolean first = true;
				for (int j=0; j<possibleRoles.length; j++) {
					if (first)
						first = false;
					else
						output.print(" && \n\t\t\t\t    session.getActiveRole() != ");
					output.print(parser.getInterfaceName() + "." + possibleRoles[j].toUpperCase());
				}
				output.println(")");
				output.println("\t\t\t\t\tISOException.throwIt(ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED);");
			}

			//Handle parameters
			Parameter[] parameters = methods[i].getParameters();
			for (int modifier=0; modifier<4; modifier++) {
				boolean mustBeConfidential = modifier==1 || modifier==2;
				boolean mustBeAuthentic = modifier==2 || modifier==3;
				for (int j=0; j<parameters.length; j++)
					if (parameters[j].isConfidential()==mustBeConfidential && parameters[j].isAuthentic()==mustBeAuthentic) {
						output.print("\t\t"+ parameters[j].getType() + " " + parameters[j].getName() + " = ");
						if (parameters[j].getType().equals("byte"))
							output.println("session.readByte();");
						else if (parameters[j].getType().equals("boolean"))
							output.println("session.readBoolean();");
						else if (parameters[j].getType().equals("short"))
							output.println("session.readShort();");
						else if (parameters[j].getType().equals("byte[]")) {
							output.println("JCSystem.makeTransientByteArray(session.readByteArrayLength(), JCSystem.CLEAR_ON_RESET);");
							output.println("\t\tsession.readByteArray(" + parameters[j].getName() + ", (short) 0);");
						}
					}
			}

			//Call implementation
			if (!methods[i].getType().equals("void"))
				output.println("\t\t\t\t" + methods[i].getType() + " result;");
			output.println("\t\t\t\tbyte exception = (byte) 0;");
			String[] exceptions = methods[i].getExceptions();
			if (exceptions != null && exceptions.length > 0) {
				output.println("\t\t\t\ttry {");
				output.print("\t");
			}
			output.print("\t\t\t\t");
			if (!methods[i].getType().equals("void"))
				output.print("result = ");
			output.print(parser.getInterfaceName().toLowerCase() + "." + methods[i].getName() + "(");
			boolean first = true;
			for (int j=0; j<parameters.length; j++) {
				if (first)
					first = false;
				else
					output.print(", ");
				output.print(parameters[j].getName());
			}
			output.println(");");

			//Calculate outgoingBytes
			String type = methods[i].getType();
			if (type.equals("byte[]")) {
				output.println("\t\t\t\\ttoutgoingBytes = JCSystem.makeTransientByteArray((short) (result.length+1), JCSystem.CLEAR_ON_RESET);");
				output.println("\t\t\t\t\tUtil.arrayCopy(result, (short) 0, outgoingBytes, (short) 1, (short) result.length);");
			}
			else if (type.equals("byte")) {
				output.println("\t\t\t\t\toutgoingBytes = JCSystem.makeTransientByteArray((short) 2, JCSystem.CLEAR_ON_RESET);");
				output.println("\t\t\t\t\toutgoingBytes[1] = result;");
			}
			else if (type.equals("short")) {
				output.println("\t\t\t\t\toutgoingBytes = JCSystem.makeTransientByteArray((short) 3, JCSystem.CLEAR_ON_RESET);");
				output.println("\t\t\t\t\tUtil.setShort(outgoingBytes, (short) 1, result);");
			}
			else if (type.equals("boolean")) {
				output.println("\t\t\t\t\toutgoingBytes = JCSystem.makeTransientByteArray((short) 2, JCSystem.CLEAR_ON_RESET);");
				output.println("\t\t\t\t\toutgoingBytes[1] = (boolean) result;");
			}
			else if (type.equals("void"))
				output.println("\t\t\t\t\toutgoingBytes = JCSystem.makeTransientByteArray((short) 1, JCSystem.CLEAR_ON_RESET);");

			//Catch exceptions
			if (exceptions != null && exceptions.length > 0) {
				output.println("\t\t\t\t}");
				for (int j=0; j<exceptions.length; j++)
					output.println("\t\t\t\tcatch (" + exceptions[j] + " e" + Integer.toString(j+1) + ") { exception = " + Integer.toString(j+1) + "; }");
			}
						output.println("\t\t\t\toutgoingBytes[0] = exception;");

			output.println("\t\t\t\tbreak;");
			output.println("\t\t\t}");
		}
		output.println("\t\t}");
		output.println("\t}");
		output.println();

		output.println("\tpublic static void install(byte[] bArray, short bOffset, byte bLength) throws ISOException {");
		output.println("\t\tnew " + parser.getInterfaceName() + "Skeleton();");
		output.println("\t}");
		output.println();

		output.println("\tpublic void process(APDU apdu) {");
		output.println("\t\tISOException.throwIt(ISO7816.SW_APPLET_SELECT_FAILED); //This applet should not be selected directly");
		output.println("\t}");
		output.println();

		output.println("\tpublic void setIncomingSize(short size) {");
		output.println("\t\tincomingBytes = JCSystem.makeTransientByteArray(size, JCSystem.CLEAR_ON_RESET);");
		output.println("\t}");
		output.println();

		output.println("\tpublic short getOutgoingSize() {");
		output.println("\t\treturn outgoingBytes==null ? (short) 0 : (short) outgoingBytes.length;");
		output.println("\t}");
		output.println();

		output.println("\tpublic void putByte(short index, byte value) {");
		output.println("\t\tincomingBytes[index] = value;");
		output.println("\t}");
		output.println();

		output.println("\tpublic byte getByte(short index) {");
		output.println("\t\treturn outgoingBytes[index];");
		output.println("\t}");
		output.println();

		output.println("\tpublic Shareable getShareableInterfaceObject(AID clientAID, byte parameter) {");
		output.println("\t\treturn parameter==1 ? (Shareable) session : (Shareable) this;");
		output.println("\t}");

		output.println("}");
		output.close();
	}

	public Main(String[] args) throws IOException {
		String jasonFilename = args[args.length-1];
		if (!jasonFilename.toLowerCase().endsWith(".jason"))
			throw new IOException(jasonFilename + " is not a .jason file");
		parser = new Parser(new File(jasonFilename));
		try {
			parser.parse();
		}
		catch (ParseException e) {
			System.out.println("ParseException at " + e.getErrorOffset() + ": " + e.getMessage());
		}
		jasonFile = new File(jasonFilename);
		String filename = jasonFile.getName().substring(0, jasonFile.getName().indexOf(".jason"));
		packageDir = new File(jasonFile.getParent());
		interfaceFile = new File(packageDir, filename + ".java");
		stubFile = new File(packageDir, filename + "Impl_Stub.java");
		skeletonFile = new File(packageDir, filename + "_Skel.java");
		makeInterface();
		makeStub();
		makeSkeleton();
	}

	private static final short computeShortHash(String s)	{
		short hash = 0;
		try	{
			MessageDigest sha = MessageDigest.getInstance("SHA");
			byte value[] = s.getBytes("UTF-8");
			byte hasharray[] = sha.digest(value);
			if(hasharray.length >= 2)
				hash = (short)(hasharray[0] << 8 | hasharray[1] & 0xff);
			else
				hash = -1;
		}
		catch (IOException ioe) {
			hash = -1;
		}
		catch(NoSuchAlgorithmException complain) {
			throw new SecurityException(complain.getMessage());
		}
		return hash;
	}

	private String getDescriptor(Method method) {
		Parameter[] params = method.getParameters();
		StringBuffer descriptor = new StringBuffer("(");
		for(int i = 0; i < params.length; i++) {
			String pName = params[i].getType();
			if (pName.endsWith("[]")) {
				pName = pName.substring(0, pName.length()-2);
				descriptor.append("[");
			}
			if("boolean".equals(pName))
				descriptor.append("Z");
			else if("byte".equals(pName))
				descriptor.append("B");
			else if("short".equals(pName))
				descriptor.append("S");
			else if("int".equals(pName))
				descriptor.append("I");
			else
				descriptor.append(pName);
		}

		descriptor.append(")");
		String rName = method.getType();
		if (rName.endsWith("[]")) {
			rName = rName.substring(0, rName.length()-2);
			descriptor.append("[");
		}
		if("boolean".equals(rName))
			descriptor.append("Z");
		else if("byte".equals(rName))
			descriptor.append("B");
		else if("short".equals(rName))
			descriptor.append("S");
		else if("int".equals(rName))
			descriptor.append("I");
		else if("void".equals(rName))
			descriptor.append("V");
		else if("[Z".equals(rName))
			descriptor.append("[Z");
		else if("[B".equals(rName))
			descriptor.append("[B");
		else if("[S".equals(rName))
			descriptor.append("[S");
		else if("[I".equals(rName)) {
			descriptor.append("[I");
		}	else {
			descriptor.append("L");
			descriptor.append(rName.replace('.', '/'));
			descriptor.append(";");
		}
		return descriptor.toString();
	}

	public int typeToValue(String type) {
		int result;
		if (type.startsWith("byte"))
			result = TYPE_BYTE;
		else if (type.startsWith("boolean"))
			result = TYPE_BOOLEAN;
		else if (type.startsWith("short"))
			result = TYPE_SHORT;
		else if (type.startsWith("int"))
			result = TYPE_INT;
		else if (type.startsWith("void"))
			result = TYPE_VOID;
		else
			result = TYPE_OBJECT;
		if (type.endsWith("[]"))
			result |= TYPE_ARRAY;
		return result;
	}

	public static void main(String[] args) throws IOException {
		if (args.length == 0 || args[args.length-1].substring(0, 1).equals("-")) {
			System.out.println("Syntax: java jason.compiler.Main [options] <jasonfile>");
			System.out.println("where options are not yet implemented");
		} else {
			Main compiler = new Main(args);
		}
	}
}