package jason.client;

import java.io.*;
import java.security.Key;
import java.util.*;
import javacard.security.KeyBuilder;

/**
 * <p>Title: Javacards As Secure Object Network</p>
 * <p>Description: KeyStore to be used on a client side computer</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: University of Twente</p>
 * @author Richard Brinkman
 * @version 1.0
 */
public class KeyStore implements Serializable {
	/** List of all supported algorithm names */
	protected static final String[] algorithmNames = {"DES"};
	/** List of all supported algorithm types */
	protected static final byte[] algorithmValues = {KeyBuilder.TYPE_DES};

	/** Mapping between role byte and Key */
	protected Hashtable keys;
	/** Mapping between role byte and the name of the corresponding session algorithm name */
	protected Hashtable sessionAlgorithms;
	protected Hashtable freshnessCounters;

	/** Initializes a new KeyStore. */
	public KeyStore() {
		keys = new Hashtable();
		sessionAlgorithms = new Hashtable();
		freshnessCounters = new Hashtable();
	}

	/**
	 * Gets the Key corresponding to the role.
	 * @param role Role byte
	 * @return The corresponding Key
	 */
	public Key getKey(byte role) {
		return (Key) keys.get(new Byte(role));
	}

	/**
	 * Sets a new key.
	 * @param role Role byte
	 * @param key The key
	 * @param sessionAlgorithm The name of the algorithm used encrypting the whole session
	 */
	public void setKey(byte role, Key key, String sessionAlgorithm) {
		keys.put(new Byte(role), key);
		sessionAlgorithms.put(new Byte(role), sessionAlgorithm);
	}

	/**
	 * Mapping between algorithm names and its value
	 * @param algorithmName The name of the algorithm
	 * @return The value of the algorithm
	 */
	public static byte algorithmNameToValue(String algorithmName) {
		for (int i=0; i<algorithmNames.length; i++)
			if (algorithmNames[i].equals(algorithmName))
				return algorithmValues[i];
		return (byte) -1;
	}

	/**
	 * Mapping between the value of an algorithm and it's name.
	 * @param algorithmValue The value of the algorithm
	 * @return The name of the algorithrm
	 */
	public static String algorithmValueToName(byte algorithmValue) {
		for (int i=0; i<algorithmValues.length; i++)
			if (algorithmValues[i] == algorithmValue)
				return algorithmNames[i];
		return null;
	}

	/**
	 * Gets the algorithm value of the key associated with the given role.
	 * @param role The role byte
	 * @return The algorithm value
	 */
	public byte getAlgorithmValue(byte role) {
		return algorithmNameToValue(getKey(role).getAlgorithm());
	}

	/**
	 * Gets the algorithm name of the session key that should be used when
	 * logging in as the specified role.
	 * @param role The role byte
	 * @return The session algorithm name
	 */
	public String getSessionAlgorithmName(byte role) {
		return (String) sessionAlgorithms.get(new Byte(role));
	}

	/**
	 * Gets the algorithm value of the session key that should be used when
	 * logging in as the specified role.
	 * @param role The role byte
	 * @return The session algorithm value
	 */
	public byte getSessionAlgorithmValue(byte role) {
		return algorithmNameToValue(getSessionAlgorithmName(role));
	}

	/**
	 * Increases the freshness counter for the specified role. The freshness
	 * counter should be increased each time a signature is calculated.
	 * @param role
	 */
	public void increaseFreshnessCounter(byte role) {
		freshnessCounters.put(new Byte(role), new Byte((byte) (getFreshnessCounter(role)+1)));
	}

	/**
	 * Gets the freshness counter for the specified role.
	 * @param role The role byte for which the freshness counter is requested
	 * @return The freshness counter
	 */
	public byte getFreshnessCounter(byte role) {
		Byte freshnessCounter = (Byte) freshnessCounters.get(new Byte(role));
		if (freshnessCounter == null) {
			freshnessCounter = new Byte((byte) 0);
			freshnessCounters.put(new Byte(role), freshnessCounter);
		}
		return freshnessCounter.byteValue();
	}

  /**
   * Loads the key store from an InputStream. The stream will be closed.
   * @param input The inputstream
   * @return A key store instance
   * @throws IOException when the input stream cannot be read from
   */
  public static KeyStore load(InputStream input) throws IOException {
    KeyStore keyStore = null;
 		try {
			ObjectInputStream objectInput = input instanceof ObjectInputStream ? (ObjectInputStream) input : new ObjectInputStream(input);
			keyStore = (KeyStore) objectInput.readObject();
			objectInput.close();
		}
		catch (ClassNotFoundException cnfe) {
			cnfe.printStackTrace();
		}
    return keyStore;
  }

  /**
   * Stores the key store to an OutputStream. The stream will be closed.
   * @param output The outputstream
   * @throws IOException when the output stream cannot be written to
   */
  public void store(OutputStream output) throws IOException {
    ObjectOutputStream objectOutput = output instanceof ObjectOutputStream ? (ObjectOutputStream) output : new ObjectOutputStream(output);
    objectOutput.writeObject(this);
    objectOutput.close();
  }

	/**
	 * Shows the contents of the keystore.
	 */
	public void show() {
		System.out.println("Keys:");
		Enumeration enumeration = keys.keys();
		while (enumeration.hasMoreElements()) {
			Byte role = (Byte) enumeration.nextElement();
			System.out.print(role + " --> ");
			Key key = (Key) keys.get(role);
			if (key instanceof XORKey)
				System.out.println(arrayToString(((XORKey) key).getEncoded()));
			if (key instanceof XORPrivateKey)
				System.out.println(arrayToString(((XORPrivateKey) key).getEncoded()));
			if (key instanceof XORPublicKey)
				System.out.println(arrayToString(((XORPublicKey) key).getEncoded()));
		}

		System.out.println("sessionAlgorithms:");
		enumeration = sessionAlgorithms.keys();
		while (enumeration.hasMoreElements()) {
			Byte role = (Byte) enumeration.nextElement();
			System.out.println(role + " --> " + sessionAlgorithms.get(role));
		}

		System.out.println("freshnessCounters:");
		enumeration = freshnessCounters.keys();
		while (enumeration.hasMoreElements()) {
			Byte role = (Byte) enumeration.nextElement();
			System.out.println(role + " --> " + freshnessCounters.get(role));
		}
	}

	private static String arrayToString(byte[] data) {
		StringBuffer buffer = new StringBuffer();
		if (data != null)
			for (int i=0; i<data.length; i++) {
				String str = Integer.toHexString(data[i] & 0xFF);
				if (str.length() < 2)
					buffer.append('0');
				buffer.append(str);
				buffer.append(' ');
			}
		return buffer.toString();
	}
}