package jason.compiler;

import java.io.*;
import java.text.ParseException;
import java.util.Vector;

public class Parser {
	private String packageName;
	private String interfaceName;
	private Vector roles;
	private Vector methods;
	private Vector imports;
	
	String contents;
	String strippedFromComments;
	int parseIndex;
	int line;
	
	private Parser() {
		roles = new Vector();
		methods = new Vector();
	}
	
	public Parser(File file) throws IOException {
		this();
		StringBuffer buffer = new StringBuffer();
		BufferedReader reader = new BufferedReader(new FileReader(file));
		String line;
		while ((line=reader.readLine()) != null)
			buffer.append(line + "\n");
		contents = buffer.toString();
	}

	public Parser(String contents) {
		this();
		this.contents = contents;
	}
	
	public void parse() throws ParseException {
		parseIndex = 0;
		line = 1;
		String token;
		stripComments();
		token = nextToken();
		if (token.equals("package")) {
			packageName = nextToken();
			if (!nextToken().equals(";"))
				throw new ParseException("Semicolon expected", line);
			token = nextToken();
		} 
		while (token.equals("import")) {
			if (imports == null)
				imports = new Vector();
			imports.add(nextToken());
			if (!nextToken().equals(";"))
				throw new ParseException("Semicolon expected, found: " + token, line);
			token = nextToken();
		}
		if (token.equals("public")) {
			token = nextToken();
			if (token.equals("interface")) {
				interfaceName = nextToken();
				if (!nextToken().equals("{"))
					throw new ParseException("{ expected", line);
				token = nextToken();
				while (!token.equals("}")) {
					if (token.equals("roles")) {
						roles.add(nextToken());
						token = nextToken();
						while (token.equals(",")) {
							roles.add(nextToken());
							token = nextToken();
						}
						if (!token.equals(";"))
							throw new ParseException("semicolon expected", line);
						token = nextToken();
					} else if (!token.equals("}")) {
						Method method = new Method();
						if (token.equals("accessible")) {
							if (!nextToken().equals("to"))
								throw new ParseException("to expected", line);
							token = nextToken();
							method.addRole(token);
							token = nextToken();
							while (token.equals(",")) {
								token = nextToken();
								method.addRole(token);
								token = nextToken();
							}
						}
						if (!token.equals("public"))
							throw new ParseException("public expected, found:" + token, line);
						token = nextToken();
						while (token.equals("authentic") || token.equals("confidential")) {
							if (token.equals("authentic")) 
								method.setAuthentic();
							else if (token.equals("confidential")) 
								method.setConfidential();
							token = nextToken();
						}	
						method.setType(token);
						method.setName(nextToken());
						if (!nextToken().equals("("))
							throw new ParseException("( expected", line);
						token = nextToken();
						while (!token.equals(")")) {
							Parameter parameter = new Parameter();
							while (token.equals("authentic") || token.equals("confidential")) {
								if (token.equals("authentic"))
									parameter.setAuthentic();
								else if (token.equals("confidential"))
									parameter.setConfidential();
								token = nextToken();
							}
							parameter.setType(token);
							parameter.setName(nextToken());
							method.addParameter(parameter);
							token = nextToken();
							if (token.equals(","))
								token = nextToken();
							else if (!token.equals(")"))
								throw new ParseException(") expected", line);
						}
						token = nextToken();
						if (token.equals("throws")) {
							method.addException(nextToken());
							token = nextToken();
							while (!token.equals(";")) {
								method.addException(nextToken());
								token = nextToken();
							}
						}
						if (!token.equals(";"))
							throw new ParseException("semicolon expected", line);
						token = nextToken();
						methods.add(method);
					}
				}
			} else
				throw new ParseException("interface expected, found: " + token, line);
		} else
			throw new ParseException("public expected, found: " + token, line);
	}

	private String nextToken() {
		skipWhiteSpace();
		if (parseIndex==strippedFromComments.length())
			return null;
		StringBuffer buffer = new StringBuffer();
		boolean tokenNotFound = true;
		char ch = strippedFromComments.charAt(parseIndex);
		if (ch==';' || ch=='(' || ch==')' || ch=='{' || ch=='}' || ch==',') {
			parseIndex++;
			return "" + ch;
		}
		if (ch=='\"') {
			int quoteIndex = strippedFromComments.indexOf("\"", parseIndex+1)+1;
			String str = strippedFromComments.substring(parseIndex, quoteIndex);
			parseIndex = quoteIndex;
			return str;
		}
		while (parseIndex < strippedFromComments.length() && tokenNotFound) {
			ch = strippedFromComments.charAt(parseIndex++);
			if (ch==' ' || ch=='\t' || ch=='\n' || ch=='(' || ch==')' || ch=='{' || ch=='}' || ch==';' || ch==',') {
				parseIndex--;
				tokenNotFound = false;
			} else
				buffer.append(ch);
		}
		return buffer.toString();
	}
	
	private void skipWhiteSpace() {
		char ch;
		boolean stop = false;
		while (!stop && parseIndex < strippedFromComments.length()) {
			ch = strippedFromComments.charAt(parseIndex);
			if (ch=='\n')
				line++;
			if (ch==' ' || ch=='\t' || ch=='\n')
				parseIndex++;		
			else
				stop = true;
		}
	}
	
	private void stripComments() {
		StringBuffer buffer = new StringBuffer();
		boolean blockComment = false;
		boolean lineComment = false;
		char ch, nextch;
		for (int i=0; i<contents.length(); i++) {
			ch = contents.charAt(i);
			switch (ch) {
				case '/' :
					if (i<contents.length()-1) {
						nextch = contents.charAt(i+1);
						switch (nextch) {
							case '/': lineComment = true; i++; break;
							case '*': blockComment = true; i++; break;
							default: 
								if (!blockComment && !lineComment)
									buffer.append(ch);
						}
					}
					break;
				case '*':
					if (i<contents.length()-1) 
						if (contents.charAt(i+1)=='/') {
							blockComment = false;
							i++;
						} else if (!blockComment && !lineComment) 
							buffer.append(ch);
					break;
				case '\n': lineComment = false; if (!blockComment) buffer.append(ch); break;
				default: 
					if (!blockComment && !lineComment)
						buffer.append(ch);
			}
		}
		strippedFromComments = buffer.toString();
	}
	
	public String getPackageName() {
		return packageName;
	}
	
	public String getInterfaceName() {
		return interfaceName;
	}
	
	public String[] getRoles() {
		if (roles == null)
			return null;
		String[] result = new String[roles.size()];
		roles.toArray(result);
		return result;
	}
	
	public Method[] getMethods() {
		if (methods == null)
			return null;
		Method[] result = new Method[methods.size()];
		methods.toArray(result);
		return result;
	}
	
	public String[] getImports() {
		if (imports == null)
			return null;
		String[] result = new String[imports.size()];
		imports.toArray(result);
		return result;
	}
	
	public static String removeQuotes(String str) {
		if (str.startsWith("\"") && str.endsWith("\""))
			return str.substring(1, str.length()-1);
		else
			return str;
	}

	/**
	 * Reads the whole contents of an {@link java.io.InputStream InputStream} into a {@link String}.
	 *
	 * @param inputStream The InputStream
	 * @return The whole string
	 * @throws IOException
	 */
	public static String readString(InputStream inputStream) throws IOException {
		StringBuffer result = new StringBuffer();
		BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
		String line;
		while ((line=reader.readLine()) != null)
			result.append(line + "\n");
		return result.toString();
	}
}