package jason.client;

import com.sun.javacard.javax.smartcard.rmiclient.JavaCardRMIConnect;
import java.io.*;
import java.rmi.RemoteException;
import java.util.Properties;

/**
 * <p>Title: Jason As Secure Object Network</p>
 * <p>Description: Applet Name Server</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: University of Twente</p>
 * @author Richard Brinkman
 * @version 1.0
 */
public class Ans {
	protected Properties properties;
	protected KeyStore keyStore;
	protected SecureCardAccessor cardAccessor;
	protected JavaCardRMIConnect javaCardRMIConnect;

	/**
	 * Loads the default properties from jcrmi/client/ans.properties.
	 */
	public Ans(KeyStore keyStore) {
		properties = new Properties();
		try {
			properties.load(ClassLoader.getSystemResourceAsStream("jason/client/ans.properties"));
			init(keyStore);
		}
		catch (IOException e) {
			e.printStackTrace();
			keyStore = null;
			cardAccessor = null;
			javaCardRMIConnect = null;
		}
	}

	/**
	 * Sets the properties to a custom value.
	 * @param keyStore The key store to be used
	 * @param properties Properties to be used
	 */
	public Ans(KeyStore keyStore, Properties properties) {
		this.properties = properties;
		init(keyStore);
	}

	/**
	 * Initializes the {@link #keyStore}, {@link #cardAccessor} and
	 * {@link #javaCardRMIConnect}.
	 */
	protected void init(KeyStore keyStore) {
		this.keyStore = keyStore;
		cardAccessor = new SecureCardAccessor();
		try {
			cardAccessor.setRemote(properties.getProperty("host", "localhost"), Integer.parseInt(properties.getProperty("port", "8080")));
			System.out.println("RMI connecting....");
			javaCardRMIConnect = new JavaCardRMIConnect(cardAccessor);
		}
		catch (RemoteException e) {
			System.out.println("Cannot connect to host");
		}
	}

	/**
	 * Gets the AID from the <code>fullAppletName</code> and selects the Applet.
	 * The it logs in as the specified <code>role</code> and gets the
	 * initial reference to the remote object.
	 * @param fullAppletName Name of the card object preceded with the full package name
	 * @param role Role byte in which to log in
	 * @return The initial reference to the remote object
	 * @throws LoginException when the card refuses to log in
	 * @throws IOException when another IO error occurs
	 */
	public Object getApplet(String fullAppletName, byte role) throws IOException {
		System.out.println("selectApplet");
		javaCardRMIConnect.selectApplet(getAppletAID(fullAppletName));
		if (cardAccessor.login(role, keyStore))
			System.out.println("Logged in");
		else {
			System.out.println("Not logged in");
			throw new LoginException("Could not log in as role " + role);
		}
		System.out.println("\ngetInitialReference");
		return javaCardRMIConnect.getInitialReference();
	}

	/**
	 * Sets the hostname.
	 * @param hostName The host name
	 */
	public void setHost(String hostName) {
		properties.setProperty("host", hostName);
	}

	/**
	 * Sets the port number.
	 * @param port The port number
	 */
	public void setPort(int port) {
		properties.setProperty("port", Integer.toString(port));
	}

	/**
	 * Loads the properties from the given input stream. The stream will beclosed.
	 * @param properties InputStream from where the properties can be read
	 * @throws IOException when an IO error occurs
	 */
	public void load(InputStream properties) throws IOException {
		this.properties.load(properties);
		properties.close();
	}

	/**
	 * Stores the properties to the given output stream. The stream will be closed.
	 * @param properties OutputStream to where the properties can be written
	 * @throws IOException when an IO error occurs
	 */
	public void store(OutputStream properties) throws IOException {
		this.properties.store(properties, null);
		properties.close();
	}

	/**
	 * Adds a new applet to the registry.
	 * @param fullAppletName Applet name preceded by the full package name
	 * @param aid The aid in the form 0x00:0x00:0x00:0x00:0x00
	 */
	public void addApplet(String fullAppletName, String aid) {
		properties.setProperty(fullAppletName, aid);
	}

	/**
	 * Adds a new applet to the registry.
	 * @param fullAppletName Applet name preceded by the full package name
	 * @param aid The aid in the form of a byte array
	 */
	public void addApplet(String fullAppletName, byte[] aid) {
		StringBuffer buffer = new StringBuffer();
		boolean first = true;
		for (int i=0; i<aid.length; i++) {
			if (!first) {
				buffer.append(':');
				first = false;
			}
			buffer.append("0x");
			buffer.append(Integer.toHexString(aid[i] & 0xFF));
		}
		addApplet(fullAppletName, buffer.toString());
	}

	/**
	 * Deletes the applet from the registry.
	 * @param fullAppletName The applet name preceded by the full package name
	 */
	public void deleteApplet(String fullAppletName) {
		properties.remove(fullAppletName);
	}

	/**
	 * Gets the AID of an applet as a byte array.
	 * @param fullAppletName The applet name preceded by the full package name
	 * @return AID
	 */
	public byte[] getAppletAID(String fullAppletName) {
		String aidString = properties.getProperty(fullAppletName);
		int count = 1;
		for (int i=0; i<aidString.length(); i++)
			count += aidString.charAt(i) == ':' ? 1 : 0;
		byte[] aid = new byte[count];
		int offset = 0;
		for (int i=0; i<count; i++) {
			String value;
			if (i==count-1)
				value = aidString.substring(offset);
			else
				value = aidString.substring(offset, aidString.indexOf(':', offset));
			if (value.startsWith("0x"))
				aid[i] = (byte) Integer.parseInt(value.substring(2), 16);
			else
				aid[i] = (byte) Integer.parseInt(value);
			offset = aidString.indexOf(':', offset+1) + 1;
		}
		return aid;
	}

	static {
		System.loadLibrary("NativeMethods");
	}
}