package jason.server;

import javacard.framework.*;
import javacard.framework.service.BasicService;
import javacard.security.*;
import javacardx.crypto.Cipher;
import jason.Constants;

/**
 * <p>Title: Javacards As Secure Object Store</p>
 * <p>Description: Session class containing methods to set up a secure session between client and server.</p>
 * <p>Copyright: Copyright (c) 2002</p>
 * <p>Company: University of Twente</p>
 * @author Richard Brinkman
 * @version 1.0
 */
public class Session extends BasicService implements Constants {
	/** INS used for a method invocation */
	public static final byte INS_INVOKE = (byte) 0x38;
	/** INS used for loggin in */
	public static final byte INS_LOGIN = (byte) 0x39;
	/** INS used for personalization */
	public static final byte INS_PUT_KEY = (byte) 0x40;

	public static final byte ROLE_CARD = (byte) 0x00;

	private RandomData randomData;
	private byte[] clientRandom;
	private byte[] serverRandom;
	private byte role;
	private boolean failure;

	private KeyStore keyStore;
	private Cipher cipher;
	private Signature signature;
	private XORKey sessionKey;

	/**
	 * The jdf array contains all information from the Jason Definition File.
	 * It has the following format:
	 * <pre><code>
	 *   jdf {
	 *     u1 <number of methods>
	 *     method[] methods
	 *   }
	 *   method {
	 *     u2 <method number>
	 *     u1 <number of roles>
	 *     u1[] roles
	 *     u1 <return modifier>
	 *     u1 <number of parameters>
	 *     u1[] modifier
	 *   }
	 * </code></pre>
	 */
	private byte[] jdf;
	/**
	 * 	Set by {@link #decrypt} to the index within the {@link #jdf} array where
	 *  the invoked method starts
	 */
	private short methodOffset;

	private boolean isInvoking;
	private boolean isLoggingIn;

	/**
	 * Default constructor
	 * @param keyStore The KeyStore that will be used with this Session object
	 * @param jdf The specific security requirements for a single Object
	 */
	public Session(KeyStore keyStore, byte[] jdf) {
		this.keyStore = keyStore;
		this.jdf = jdf;
		isInvoking = false;
		isLoggingIn = false;
		randomData = RandomData.getInstance(RandomData.ALG_SECURE_RANDOM);
		clientRandom = new byte[8];
		serverRandom = new byte[8];
	}

	/**
	 * Remembers if the apdu is an invoke APDU. The value of {@link #isInvoking} is
	 * used in the {@link #processDataOut} method to decide if the outgoing APDU
	 * should be encrypted or not. If it is an invoke APDU the {@link #decrypt}
	 * method is invoked
	 * @param apdu The unprocessed APDU
	 * @return false or the return value of {@link #decrypt}
	 */
	public boolean processDataIn(APDU apdu) {
		failure = false;
		if (getINS(apdu) == INS_INVOKE) {
			isInvoking = true;
			return decrypt(apdu);
		} else {
			isInvoking = false;
			return false;
		}
	}

	/**
	 * The Session object itself can also process some commands. Login and
	 * personalization are the only commands that the
	 * Session will process. While encountering an {@link #INS_LOGIN} the
	 * {@link #login login} method is called. While encountering an
	 * {@link #INS_PUT_KEY} the {@link #putKey putKey} is called. All
	 * other INS values will keep the <code>apdu</code> unprocessed.
	 * @param apdu
	 * @return true if the apdu should not further be processed, false otherwise
	 */
	public boolean processCommand(APDU apdu) {
		if (failure)
			return true;
		switch (getINS(apdu)) {
			case INS_LOGIN:
				return login(apdu);
			case INS_PUT_KEY:
				return putKey(apdu);
			default:
				return false;
		}
	}

	/**
	 * For each login procedure the <code>login</code> method will be invoked
	 * twice. The first time (when {@link #isLoggingIn} is false) it will receive
	 * an APDU in the following format:
	 * <table border="1" cellspacing="0">
	 *   <tr>
	 *     <td>CLA</td>
	 *     <td>INS</td>
	 *     <td>P1</td>
	 *     <td>P2</td>
	 *     <td>Lc</td>
	 *     <td>Role<br>byte</td>
	 *     <td>ClientRandom<br>length</td>
	 *     <td>ClientRandom</td>
	 *   </tr>
	 * </table>
	 * The response APDU that is being sent the first time is:
	 * <table border="1" cellspacing="0">
	 *   <tr>
	 *     <td>SW1</td>
	 *     <td>SW2</td>
	 *     <td>ServerRandom<br>length</td>
	 *     <td>ServerRandom</td>
	 *     <td>SignedClientRandom<br>length</td>
	 *     <td>SignedClientRandom</td>
	 *   </tr>
	 * </table>
	 * The second time the <code>login</code> method is invoked, it expects an
	 * APDU of the following format:
	 * <table border="1" cellspacing="0">
	 *   <tr>
	 *     <td>CLA</td>
	 *     <td>INS</td>
	 *     <td>P1</td>
	 *     <td>P2</td>
	 *     <td>Lc</td>
	 *     <td>SignedServerRandom<br>length</td>
	 *     <td>SignedServerRandom</td>
	 *   </tr>
	 * </table>
	 * The second time the response APDU will look like:
	 * <table border="1" cellspacing="0">
	 *   <tr>
	 *     <td>SW1</td>
	 *     <td>SW2</td>
	 *     <td>EncryptedSessionKey<br>length</td>
	 *     <td>EncryptedSessionKey</td>
	 *   </tr>
	 * </table>
	 * @param apdu unprocessed APDU
	 * @return true if the apdu can be considered processed
	 */
	private boolean login(APDU apdu) {
		byte[] buffer = apdu.getBuffer();
		if (!isLoggingIn) {
			signature = Signature.getInstance((byte) 0, false);
			isLoggingIn = true;
			apdu.setIncomingAndReceive();
			role = buffer[5];
			Util.arrayCopy(buffer, (short) 7, clientRandom, (short) 0, (short) 8);
			isInvoking = false;
			apdu.setOutgoing();
			setOutputLength(apdu, (short) 18);
			randomData.setSeed(clientRandom, (short) 0, (short) 8);
			randomData.generateData(serverRandom, (short) 0, (short) 8);
			short offset = (short) 5;
			buffer[offset++] = (byte) serverRandom.length;
			Util.arrayCopy(serverRandom, (short) 0, buffer, offset, (short) serverRandom.length);
			offset += (short) serverRandom.length;
			buffer[offset++] = (byte) 8;
			XORPrivateKey key = (XORPrivateKey) keyStore.getKey(ROLE_CARD);

			signature.init(key, Signature.MODE_SIGN);
			signature.sign(clientRandom, (short) 0, (short) 8, buffer, offset);
			return succeed(apdu);
		} else {
			isLoggingIn = false;
			apdu.setIncomingAndReceive();
			short length = (short) buffer[5];
			if (length > 0) {
				signature.init(keyStore.getKey(role), Signature.MODE_VERIFY);
				apdu.setOutgoing();
				buffer[5] = role;
				if (signature.verify(serverRandom, (short) 0, (short) 8, buffer, (short) 6, (short) 8)) {
					buffer[6] = (byte) 1; //Meaning logged in
					buffer[7] = (byte) 8; //Length of encrypted session key
					byte[] sessionKeyData = JCSystem.makeTransientByteArray((short) 8, JCSystem.CLEAR_ON_RESET);
					randomData.generateData(sessionKeyData, (short) 0, (short) 8);
					sessionKey = new XORKey();
					sessionKey.setKey(sessionKeyData, (short) 0);
					setOutputLength(apdu, (short) 11);
					cipher = Cipher.getInstance((byte) 0, false);
					cipher.init(keyStore.getKey(role), Cipher.MODE_ENCRYPT);
					cipher.doFinal(sessionKeyData, (short) 0, (short) 8, buffer, (short) 8);
					cipher.init(sessionKey, Cipher.MODE_ENCRYPT);
				}
				else {
					setOutputLength(apdu, (short) 2);
					buffer[6] = (byte) 2; //Meaning cannot log in
				}
			} else {
				apdu.setOutgoing();
				setOutputLength(apdu, (short) 2);
				buffer[5] = role;
				buffer[6] = (byte) 0;
			}
			JCSystem.requestObjectDeletion();
			return succeed(apdu);
		}
	}

	private boolean putKey(APDU apdu) {
		isInvoking = false;
		byte[] buffer = apdu.getBuffer();
		short length = apdu.setIncomingAndReceive();
		short offset = (short) 5;
		byte role = buffer[offset++];
		byte sessionAlgorithm = buffer[offset++];
		byte keyType = buffer[offset++];
		byte keyLength = buffer[offset++];
		try {
			Key key;
			switch (keyType) {
				case KeyBuilder.TYPE_DES:
					key = KeyBuilder.buildKey(keyType, (short) (keyLength*8), false);
					((DESKey) key).setKey(buffer, offset);
					break;
				case (byte) 0x88: //XORPrivateKey
					key = new XORPrivateKey(buffer, offset, keyLength);
					break;
				case (byte) 0x89: //XORPublicKey
					key = new XORPublicKey(buffer, offset, keyLength);
					break;
				default:
					key = null;
				/** @todo: add more keytypes */
			}
			if (keyStore.setKey(role, key, sessionAlgorithm)) {
				setOutputLength(apdu, (short) 0);
				return succeed(apdu);
			}
			else {
				failure = true;
				return fail(apdu, ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED);
			}
		}
		catch (CryptoException e) {
			failure = true;
			return fail(apdu, ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED);
		}
	}

	/**
	 * Decrypts the apdu buffer when necessary. If the terminal has not logged in
	 * nothing will happen. If the signature is not correct, then an
	 * ISO7816.SW_SECURITY_STATUS exception is thrown.
	 * @param apdu The unencrypted apdu
	 * @return false
	 */
	private boolean decrypt(APDU apdu) {
		if (role != 0) { //Logged in
			byte[] buffer = apdu.getBuffer();
			apdu.setIncomingAndReceive();
			byte offset = (byte) 9; //Skip object_id and method_id
			byte lengthPlain = buffer[offset++];
			byte offsetPlain = offset;
			offset += lengthPlain;
			byte lengthConfidential = buffer[offset++];
			byte offsetConfidential = offset;
			offset += lengthConfidential;
			byte lengthAuthentic = buffer[offset++];
			byte offsetAuthentic = offset;
			offset += lengthAuthentic;
			byte freshnessCounter = buffer[offset++];
			byte lengthSignature = buffer[offset++];
			byte offsetSignature = offset;

			byte[] plainBuffer = JCSystem.makeTransientByteArray((short) lengthPlain, JCSystem.CLEAR_ON_RESET);
			if (lengthPlain > (byte) 0)
				Util.arrayCopy(buffer, offsetPlain, plainBuffer, (short) 0, lengthPlain);
			byte[] decryptedConfidential = JCSystem.makeTransientByteArray((short) lengthConfidential, JCSystem.CLEAR_ON_RESET);
			short lengthDecryptedConfidential = (short) 0;
			if (lengthConfidential > (byte) 0)
				lengthDecryptedConfidential = cipher.doFinal(buffer, (short) offsetConfidential, lengthConfidential, decryptedConfidential, (short) 0);
			byte[] authenticBuffer = JCSystem.makeTransientByteArray(lengthAuthentic, JCSystem.CLEAR_ON_RESET);
			if (lengthAuthentic > (byte) 0)
				Util.arrayCopy(buffer, offsetAuthentic, authenticBuffer, (short) 0, lengthAuthentic);
			byte[] signatureBuffer = JCSystem.makeTransientByteArray(lengthSignature, JCSystem.CLEAR_ON_RESET);
			if (lengthSignature > (byte) 0)
				Util.arrayCopy(buffer, offsetSignature, signatureBuffer, (short) 0, lengthSignature);

			//Check signature and unshuffle
			short jdfOffset = 0;
			byte numberOfMethods = jdf[jdfOffset++];
			boolean signaturePresent = false;
			for (short i=0; i<numberOfMethods; i++)
				if (jdf[jdfOffset] != buffer[(short) 7] || jdf[(short) (jdfOffset+1)] != buffer[(short) 8]) { //Incorrect method id
					jdfOffset += 2; //skip method_id
					jdfOffset += jdf[jdfOffset] + 2; //skip number_of_roles, roles and modifier
					jdfOffset += jdf[jdfOffset] + 1; //skip number_of_parameters and parameters
				} else {
					methodOffset = jdfOffset;
					jdfOffset += (short) 2; //Skip method_id
					byte numberOfRoles = jdf[jdfOffset++];
					boolean loggedIn = numberOfRoles == 0;
					signaturePresent = numberOfRoles != 0;
					for (short j=0; j<numberOfRoles; j++) {
						loggedIn |= jdf[jdfOffset] == role | jdf[jdfOffset] == ANYBODY;
						signaturePresent &= jdf[jdfOffset++] != ANYBODY;
					}
					byte methodModifier = jdf[jdfOffset++];
					byte numberOfParameters = jdf[jdfOffset++]; //skip number_of_parameters
//          cipher.init(sessionKey, Cipher.MODE_DECRYPT);
//          signature.init(keyStore.getKey(role), Signature.MODE_VERIFY);
					short plainOffset = (short) 0;
					short authenticOffset = (short) 0;
					short confidentialOffset = (short) 0;
					signaturePresent |= (methodModifier & SECURITY_AUTHENTIC) == SECURITY_AUTHENTIC;
					for (short j=0; j<numberOfParameters; j++)
						signaturePresent |= (jdf[(short) (jdfOffset+j)] & SECURITY_AUTHENTIC) == SECURITY_AUTHENTIC;
					if (signaturePresent) {
						signature.update(buffer, (short) 7, (short) 2); //method_id
						if ((byte) (keyStore.freshnessCounters[role]+1) != freshnessCounter) {
							failure = true;
							return fail(apdu, ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED);
						}
						keyStore.freshnessCounters[role]++;
						signature.update(keyStore.freshnessCounters, role, (short) 1);
					}
					short dataOffset = 9; //skip object_id and method_id
					for (short j=0; j<numberOfParameters; j++) {
						signaturePresent |= (jdf[(short) (jdfOffset+j)] & SECURITY_AUTHENTIC) == SECURITY_AUTHENTIC;
						short dataLength = 0;
						switch ((byte) (jdf[(short) (jdfOffset+j)] & (byte) 0x07)) {
							case TYPE_BYTE:
							case TYPE_BOOLEAN: dataLength = (short) 1; break;
							case TYPE_SHORT: dataLength = (short) 2; break;
							case TYPE_INT: dataLength = (short) 4; break;
							default: dataLength = (short) 0;
						}
						switch ((byte) (jdf[(short) (jdfOffset+j)] & (byte) 0xF0)) {
							case SECURITY_PLAIN:
								if ((byte) ((jdf[(short) (jdfOffset+j)] & TYPE_ARRAY)) == TYPE_ARRAY)
									dataLength = (short) (dataLength*plainBuffer[plainOffset] + 1);
								Util.arrayCopy(plainBuffer, plainOffset, buffer, dataOffset, dataLength);
								plainOffset += dataLength;
								break;
							case SECURITY_CONFIDENTIAL_AUTHENTIC:
								if ((byte) ((jdf[(short) (jdfOffset+j)] & TYPE_ARRAY)) == TYPE_ARRAY)
									dataLength = (short) (dataLength*decryptedConfidential[confidentialOffset] + 1);
								Util.arrayCopy(decryptedConfidential, confidentialOffset, buffer, dataOffset, dataLength);
								signature.update(decryptedConfidential, confidentialOffset, dataLength);
								confidentialOffset += dataLength;
								break;
							case SECURITY_CONFIDENTIAL:
								if ((byte) ((jdf[(short) (jdfOffset+j)] & TYPE_ARRAY)) == TYPE_ARRAY)
									dataLength = (short) (dataLength*decryptedConfidential[confidentialOffset] + 1);
								Util.arrayCopy(decryptedConfidential, confidentialOffset, buffer, dataOffset, dataLength);
								confidentialOffset += dataLength;
								break;
							case SECURITY_AUTHENTIC:
								if ((byte) ((jdf[(short) (jdfOffset+j)] & TYPE_ARRAY)) == TYPE_ARRAY)
									dataLength = (short) (dataLength*authenticBuffer[authenticOffset] + 1);
								Util.arrayCopy(authenticBuffer, authenticOffset, buffer, dataOffset, dataLength);
								signature.update(authenticBuffer, authenticOffset, dataLength);
								authenticOffset += dataLength;
								break;
						}
						dataOffset += dataLength;
					}
					jdfOffset += numberOfParameters;
				}

			if (signaturePresent && !signature.verify(null, (short) 0, (short) 0, signatureBuffer, (short) 0, lengthSignature)) {
				failure = true;
				return fail(apdu, ISO7816.SW_SECURITY_STATUS_NOT_SATISFIED);
			}

			buffer[ISO7816.OFFSET_LC] = (byte) (4 + lengthPlain + lengthDecryptedConfidential + lengthAuthentic);
		}
		JCSystem.requestObjectDeletion();
		return false;
	}

	/**
	 * Encrypts or signs the return value when necessary. The
	 * {@link #methodOffset} has already been set by the {@link #decrypt} method.
	 * @param apdu The plain result APDU
	 * @return true
	 */
	private boolean encrypt(APDU apdu) {
		byte[] buffer = apdu.getBuffer();
		short jdfOffset = (short) (methodOffset + 2); //Skip method_id
		jdfOffset += (short) (jdf[jdfOffset] + 1); //Skip roles
		byte returnModifier = jdf[jdfOffset];
		short dataLength;
		switch ((byte) (returnModifier & 0x07)) {
			case TYPE_BYTE:
			case TYPE_BOOLEAN: dataLength = (short) 1; break;
			case TYPE_SHORT: dataLength = (short) 2; break;
			case TYPE_INT: dataLength = (short) 4; break;
			default: dataLength = (short) 0;
		}
		boolean isArray = false;
		if ((byte) (returnModifier & TYPE_ARRAY) != (byte) 0) {
			dataLength = (short) (dataLength*buffer[6]+1);
			isArray = true;
		}
		byte[] unencrypted = JCSystem.makeTransientByteArray(dataLength, JCSystem.CLEAR_ON_RESET);
		Util.arrayCopy(buffer, (short) 6, unencrypted, (short) 0, dataLength);
		short outputOffset = (short) 6;
		switch ((byte) (returnModifier & 0xF0)) {
			case SECURITY_PLAIN:
				outputOffset += dataLength;
				break;
			case SECURITY_CONFIDENTIAL:
				cipher.init(sessionKey, Cipher.MODE_ENCRYPT);
				outputOffset += cipher.doFinal(unencrypted, (short) 0, dataLength, buffer, outputOffset);
				break;
			case SECURITY_CONFIDENTIAL_AUTHENTIC:
				cipher.init(sessionKey, Cipher.MODE_ENCRYPT);
				signature.init(keyStore.getKey(ROLE_CARD), Signature.MODE_SIGN);
				short confidentialLength = cipher.doFinal(unencrypted, (short) 0, dataLength, buffer, (short) (outputOffset+1));
				buffer[outputOffset] = (byte) confidentialLength;
				outputOffset += (short) (confidentialLength+1);
				keyStore.freshnessCounters[role]++;
				signature.update(keyStore.freshnessCounters, role, (short) 1);
				buffer[outputOffset++] = keyStore.freshnessCounters[role];
				outputOffset += signature.sign(unencrypted, (short) 0, dataLength, buffer, outputOffset);
				break;
			case SECURITY_AUTHENTIC:
				signature.init(keyStore.getKey(ROLE_CARD), Signature.MODE_SIGN);
				keyStore.freshnessCounters[role]++;
				signature.update(keyStore.freshnessCounters, role, (short) 1);
				outputOffset += dataLength;
				buffer[outputOffset++] = keyStore.freshnessCounters[role];
				outputOffset += signature.sign(unencrypted, (short) 0, dataLength, buffer, outputOffset);
		}
		setOutputLength(apdu, (short) (outputOffset-5));
		JCSystem.requestObjectDeletion();
		return true;
	}

	/**
	 * A select APDU will be expanded by the JDF array. An invocation APDU will
	 * be encrypted when necessary.
	 * @param apdu The APDU
	 * @return false
	 */
	public boolean processDataOut(APDU apdu) {
		if (failure)
			return true;
		byte[] buffer = apdu.getBuffer();
		if (isInvoking & buffer[2] == (byte) 0x90 && buffer[3] == (byte) 0x00 && buffer[5] == (byte) 0x81)
			return encrypt(apdu);
		else
			return false;
	}
}